<?php
// admin_packages.php
require_once 'config.php';
require_once 'contabo_api.php';

redirectIfNotLoggedIn();
redirectIfNotAdmin();

$message = '';
$error = '';

// Get current packages
$packages = ContaboAPI::getVpsPackages();

// Handle form submission for updating packages
// Handle form submission for updating packages
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $db = getDB();
        
        if (isset($_POST['update_packages'])) {
            // Update multiple packages
            $updatedPackages = [];
            
            foreach ($_POST['packages'] as $packageId => $packageData) {
                // Validate data
                if (!is_numeric($packageData['price_monthly']) || $packageData['price_monthly'] < 0) {
                    throw new Exception("Invalid price for package {$packageId}");
                }
                if (!is_numeric($packageData['price_original']) || $packageData['price_original'] < 0) {
                    throw new Exception("Invalid original price for package {$packageId}");
                }
                if (!is_numeric($packageData['cpu']) || $packageData['cpu'] < 1) {
                    throw new Exception("Invalid CPU cores for package {$packageId}");
                }
                if (!is_numeric($packageData['ram']) || $packageData['ram'] < 1) {
                    throw new Exception("Invalid RAM for package {$packageId}");
                }
                if (!is_numeric($packageData['disk']) || $packageData['disk'] < 1) {
                    throw new Exception("Invalid disk size for package {$packageId}");
                }
                if (!is_numeric($packageData['snapshots']) || $packageData['snapshots'] < 0) {
                    throw new Exception("Invalid snapshots for package {$packageId}");
                }
                
                // Build updated package array
                $updatedPackages[$packageId] = [
                    'name' => $packageData['name'],
                    'cpu' => (int)$packageData['cpu'],
                    'ram' => (int)$packageData['ram'],
                    'disk' => (int)$packageData['disk'],
                    'disk_type' => $packageData['disk_type'],
                    'snapshots' => (int)$packageData['snapshots'],
                    'traffic' => $packageData['traffic'],
                    'backup' => isset($packageData['backup']),
                    'price_monthly' => (float)$packageData['price_monthly'],
                    'price_original' => (float)$packageData['price_original'],
                    'currency' => $packageData['currency'],
                    'description' => $packageData['description']
                ];
            }
            
            // ✅ FIXED: Actually save to database
            if (ContaboAPI::savePackages($updatedPackages)) {
                $message = "Package prices and details updated successfully in database!";
                // Reload packages from database to show updated values
                $packages = ContaboAPI::getVpsPackages();
            } else {
                throw new Exception("Failed to save packages to database.");
            }
            
        } elseif (isset($_POST['add_package'])) {
            // Add new package
            $newPackageId = $_POST['new_package_id'];
            $newPackageData = $_POST['new_package'];
            
            // Validate new package
            if (empty($newPackageId)) {
                throw new Exception("Package ID is required");
            }
            
            // Load current packages
            $currentPackages = ContaboAPI::getVpsPackages();
            
            if (isset($currentPackages[$newPackageId])) {
                throw new Exception("Package ID already exists");
            }
            if (!is_numeric($newPackageData['price_monthly']) || $newPackageData['price_monthly'] < 0) {
                throw new Exception("Invalid monthly price");
            }
            
            // Add new package to current packages
            $currentPackages[$newPackageId] = [
                'name' => $newPackageData['name'],
                'cpu' => (int)$newPackageData['cpu'],
                'ram' => (int)$newPackageData['ram'],
                'disk' => (int)$newPackageData['disk'],
                'disk_type' => $newPackageData['disk_type'],
                'snapshots' => (int)$newPackageData['snapshots'],
                'traffic' => $newPackageData['traffic'],
                'backup' => isset($newPackageData['backup']),
                'price_monthly' => (float)$newPackageData['price_monthly'],
                'price_original' => (float)$newPackageData['price_original'],
                'currency' => $newPackageData['currency'],
                'description' => $newPackageData['description']
            ];
            
            // ✅ FIXED: Actually save to database
            if (ContaboAPI::savePackages($currentPackages)) {
                $message = "New package added successfully to database!";
                // Reload packages from database
                $packages = ContaboAPI::getVpsPackages();
            } else {
                throw new Exception("Failed to save new package to database.");
            }
            
        } elseif (isset($_POST['delete_package'])) {
            // Delete package
            $packageIdToDelete = $_POST['delete_package_id'];
            
            // Load current packages
            $currentPackages = ContaboAPI::getVpsPackages();
            
            if (!isset($currentPackages[$packageIdToDelete])) {
                throw new Exception("Package not found");
            }
            
            // Remove the package
            unset($currentPackages[$packageIdToDelete]);
            
            // ✅ FIXED: Actually save to database
            if (ContaboAPI::savePackages($currentPackages)) {
                $message = "Package deleted successfully from database!";
                // Reload packages from database
                $packages = ContaboAPI::getVpsPackages();
            } else {
                throw new Exception("Failed to delete package from database.");
            }
        }
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage VPS Packages - Admin</title>
    <style>
        :root {
            --primary: #007bff;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --secondary: #6c757d;
            --light: #f8f9fa;
            --dark: #343a40;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .nav-links {
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #ddd;
        }
        
        .nav-links a {
            margin-right: 15px;
            text-decoration: none;
            color: var(--primary);
        }
        
        .nav-links a:hover {
            text-decoration: underline;
        }
        
        .message {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .packages-table {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }
        
        th {
            background-color: var(--light);
            font-weight: 600;
            color: var(--dark);
        }
        
        input, select, textarea {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        input:focus, select:focus, textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 2px rgba(0,123,255,0.1);
        }
        
        .btn {
            display: inline-block;
            padding: 8px 16px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            text-decoration: none;
            font-weight: 500;
            transition: all 0.3s ease;
            text-align: center;
            font-size: 14px;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-success {
            background: var(--success);
            color: white;
        }
        
        .btn-danger {
            background: var(--danger);
            color: white;
        }
        
        .btn-warning {
            background: var(--warning);
            color: black;
        }
        
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }
        
        .btn-block {
            width: 100%;
            display: block;
        }
        
        .form-section {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: var(--dark);
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .checkbox-group input[type="checkbox"] {
            width: auto;
        }
        
        .package-preview {
            background: var(--light);
            padding: 20px;
            border-radius: 8px;
            margin-top: 20px;
        }
        
        .preview-title {
            font-weight: bold;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .action-buttons {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
            margin-top: 20px;
        }
        
        .delete-form {
            display: inline;
        }
        
        @media (max-width: 768px) {
            .form-row {
                grid-template-columns: 1fr;
            }
            
            .packages-table {
                padding: 15px;
            }
            
            table {
                font-size: 14px;
            }
            
            th, td {
                padding: 8px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="nav-links">
            <a href="view_vps.php">VPS Management</a>
            <a href="user_management.php">User Management</a>
            <a href="customer_management.php">Customer Management</a>
            <a href="admin_packages.php">Package Management</a>
            <a href="settings.php">Settings</a>
            <a href="logout.php">Logout</a>
        </div>
        
        <div class="header">
            <h1>Manage VPS Packages</h1>
            <p>Update prices, specifications, and manage packages</p>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <!-- Update Existing Packages -->
        <div class="packages-table">
            <h2>Update Existing Packages</h2>
            <form method="POST">
                <table>
                    <thead>
                        <tr>
                            <th>Package ID</th>
                            <th>Name</th>
                            <th>vCPU</th>
                            <th>RAM (GB)</th>
                            <th>Disk (GB)</th>
                            <th>Disk Type</th>
                            <th>Snapshots</th>
                            <th>Traffic</th>
                            <th>Backup</th>
                            <th>Price (Monthly)</th>
                            <th>Original Price</th>
                            <th>Currency</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($packages as $packageId => $package): ?>
                            <tr>
                                <td>
                                    <strong><?php echo htmlspecialchars($packageId); ?></strong>
                                    <input type="hidden" name="packages[<?php echo $packageId; ?>][id]" value="<?php echo $packageId; ?>">
                                </td>
                                <td>
                                    <input type="text" name="packages[<?php echo $packageId; ?>][name]" 
                                           value="<?php echo htmlspecialchars($package['name']); ?>" required>
                                </td>
                                <td>
                                    <input type="number" name="packages[<?php echo $packageId; ?>][cpu]" 
                                           value="<?php echo $package['cpu']; ?>" min="1" step="1" required>
                                </td>
                                <td>
                                    <input type="number" name="packages[<?php echo $packageId; ?>][ram]" 
                                           value="<?php echo $package['ram']; ?>" min="1" step="1" required>
                                </td>
                                <td>
                                    <input type="number" name="packages[<?php echo $packageId; ?>][disk]" 
                                           value="<?php echo $package['disk']; ?>" min="1" step="1" required>
                                </td>
                                <td>
                                    <select name="packages[<?php echo $packageId; ?>][disk_type]" required>
                                        <option value="NVMe" <?php echo $package['disk_type'] === 'NVMe' ? 'selected' : ''; ?>>NVMe</option>
                                        <option value="SSD" <?php echo $package['disk_type'] === 'SSD' ? 'selected' : ''; ?>>SSD</option>
                                    </select>
                                </td>
                                <td>
                                    <input type="number" name="packages[<?php echo $packageId; ?>][snapshots]" 
                                           value="<?php echo $package['snapshots']; ?>" min="0" step="1" required>
                                </td>
                                <td>
                                    <input type="text" name="packages[<?php echo $packageId; ?>][traffic]" 
                                           value="<?php echo htmlspecialchars($package['traffic']); ?>" required>
                                </td>
                                <td>
                                    <div class="checkbox-group">
                                        <input type="checkbox" name="packages[<?php echo $packageId; ?>][backup]" 
                                               value="1" <?php echo $package['backup'] ? 'checked' : ''; ?>>
                                        <span>Yes</span>
                                    </div>
                                </td>
                                <td>
                                    <input type="number" name="packages[<?php echo $packageId; ?>][price_monthly]" 
                                           value="<?php echo $package['price_monthly']; ?>" min="0" step="0.01" required>
                                </td>
                                <td>
                                    <input type="number" name="packages[<?php echo $packageId; ?>][price_original]" 
                                           value="<?php echo $package['price_original']; ?>" min="0" step="0.01" required>
                                </td>
                                <td>
                                    <select name="packages[<?php echo $packageId; ?>][currency]" required>
                                        <option value="€" <?php echo $package['currency'] === '€' ? 'selected' : ''; ?>>€ Euro</option>
                                        <option value="$" <?php echo $package['currency'] === '$' ? 'selected' : ''; ?>>$ USD</option>
                                        <option value="£" <?php echo $package['currency'] === '£' ? 'selected' : ''; ?>>£ GBP</option>
                                    </select>
                                </td>
                                <td>
                                    <input type="text" name="packages[<?php echo $packageId; ?>][description]" 
                                           value="<?php echo htmlspecialchars($package['description']); ?>" required>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <div class="action-buttons">
                    <button type="submit" name="update_packages" class="btn btn-success">
                        💾 Update All Packages
                    </button>
                </div>
            </form>
        </div>
        
        <!-- Add New Package -->
        <div class="form-section">
            <h2>Add New Package</h2>
            <form method="POST">
                <div class="form-row">
                    <div class="form-group">
                        <label for="new_package_id">Package ID *</label>
                        <input type="text" id="new_package_id" name="new_package_id" 
                               placeholder="V108" required pattern="[A-Za-z0-9_]+">
                        <small style="color: var(--secondary);">Unique identifier (e.g., V108)</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_name">Package Name *</label>
                        <input type="text" id="new_package_name" name="new_package[name]" 
                               placeholder="Cloud VPS 70" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_description">Description *</label>
                        <input type="text" id="new_package_description" name="new_package[description]" 
                               placeholder="Premium Performance" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="new_package_cpu">vCPU Cores *</label>
                        <input type="number" id="new_package_cpu" name="new_package[cpu]" 
                               value="4" min="1" step="1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_ram">RAM (GB) *</label>
                        <input type="number" id="new_package_ram" name="new_package[ram]" 
                               value="8" min="1" step="1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_disk">Disk Size (GB) *</label>
                        <input type="number" id="new_package_disk" name="new_package[disk]" 
                               value="100" min="1" step="1" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="new_package_disk_type">Disk Type *</label>
                        <select id="new_package_disk_type" name="new_package[disk_type]" required>
                            <option value="NVMe">NVMe</option>
                            <option value="SSD">SSD</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_snapshots">Snapshots *</label>
                        <input type="number" id="new_package_snapshots" name="new_package[snapshots]" 
                               value="1" min="0" step="1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_traffic">Traffic *</label>
                        <input type="text" id="new_package_traffic" name="new_package[traffic]" 
                               value="32 TB" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="new_package_price_monthly">Monthly Price *</label>
                        <input type="number" id="new_package_price_monthly" name="new_package[price_monthly]" 
                               value="10.00" min="0" step="0.01" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_price_original">Original Price *</label>
                        <input type="number" id="new_package_price_original" name="new_package[price_original]" 
                               value="15.00" min="0" step="0.01" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="new_package_currency">Currency *</label>
                        <select id="new_package_currency" name="new_package[currency]" required>
                            <option value="€">€ Euro</option>
                            <option value="$">$ USD</option>
                            <option value="£">£ GBP</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <div class="checkbox-group">
                        <input type="checkbox" id="new_package_backup" name="new_package[backup]" value="1">
                        <label for="new_package_backup">Include Free Auto Backup</label>
                    </div>
                </div>
                
                <div class="action-buttons">
                    <button type="submit" name="add_package" class="btn btn-primary">
                        ➕ Add New Package
                    </button>
                </div>
            </form>
        </div>
        
        <!-- Delete Package -->
        <div class="form-section">
            <h2>Delete Package</h2>
            <form method="POST" class="delete-form">
                <div class="form-row">
                    <div class="form-group" style="flex: 2;">
                        <label for="delete_package_id">Select Package to Delete *</label>
                        <select id="delete_package_id" name="delete_package_id" required>
                            <option value="">Choose a package...</option>
                            <?php foreach ($packages as $packageId => $package): ?>
                                <option value="<?php echo $packageId; ?>">
                                    <?php echo htmlspecialchars($packageId); ?> - <?php echo htmlspecialchars($package['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group" style="flex: 1; align-self: end;">
                        <button type="submit" name="delete_package" class="btn btn-danger btn-block" 
                                onclick="return confirm('⚠️ Are you sure you want to delete this package? This action cannot be undone.')">
                            🗑️ Delete Package
                        </button>
                    </div>
                </div>
            </form>
        </div>
        
        <!-- Current Packages Preview -->
        <div class="form-section">
            <h2>Current Packages Preview</h2>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;">
                <?php foreach ($packages as $packageId => $package): ?>
                    <div class="package-preview">
                        <div class="preview-title"><?php echo htmlspecialchars($package['name']); ?></div>
                        <div><strong>ID:</strong> <?php echo htmlspecialchars($packageId); ?></div>
                        <div><strong>Price:</strong> <?php echo $package['currency'] . $package['price_monthly']; ?> (was <?php echo $package['currency'] . $package['price_original']; ?>)</div>
                        <div><strong>Specs:</strong> <?php echo $package['cpu']; ?> vCPU, <?php echo $package['ram']; ?>GB RAM, <?php echo $package['disk']; ?>GB <?php echo $package['disk_type']; ?></div>
                        <div><strong>Features:</strong> <?php echo $package['snapshots']; ?> snapshots, <?php echo $package['traffic']; ?> traffic<?php echo $package['backup'] ? ', Auto Backup' : ''; ?></div>
                        <div><strong>Description:</strong> <?php echo htmlspecialchars($package['description']); ?></div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <script>
        // Auto-calculate original price based on monthly price
        document.addEventListener('DOMContentLoaded', function() {
            const monthlyPriceInputs = document.querySelectorAll('input[name$="[price_monthly]"]');
            
            monthlyPriceInputs.forEach(input => {
                input.addEventListener('input', function() {
                    const monthlyPrice = parseFloat(this.value) || 0;
                    const originalPriceInput = this.closest('tr').querySelector('input[name$="[price_original]"]');
                    
                    // Set original price 25% higher than monthly price
                    if (originalPriceInput && !originalPriceInput.dataset.manual) {
                        const originalPrice = monthlyPrice * 1.25;
                        originalPriceInput.value = originalPrice.toFixed(2);
                    }
                });
            });
            
            // Track manual changes to original price
            const originalPriceInputs = document.querySelectorAll('input[name$="[price_original]"]');
            originalPriceInputs.forEach(input => {
                input.addEventListener('input', function() {
                    this.dataset.manual = 'true';
                });
            });
            
            // New package price calculation
            const newMonthlyPrice = document.getElementById('new_package_price_monthly');
            const newOriginalPrice = document.getElementById('new_package_price_original');
            
            if (newMonthlyPrice && newOriginalPrice) {
                newMonthlyPrice.addEventListener('input', function() {
                    const monthlyPrice = parseFloat(this.value) || 0;
                    if (!newOriginalPrice.dataset.manual) {
                        const originalPrice = monthlyPrice * 1.25;
                        newOriginalPrice.value = originalPrice.toFixed(2);
                    }
                });
                
                newOriginalPrice.addEventListener('input', function() {
                    this.dataset.manual = 'true';
                });
            }
        });
    </script>
</body>
</html>