<?php
// ContaboAPI.php
require_once 'config.php';

class ContaboAPI {
    private $access_token;
    private $settings;
    
    public function __construct() {
        $this->loadSettings();
    }
    
    private function loadSettings() {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM contabo_settings LIMIT 1");
        $this->settings = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$this->settings) {
            throw new Exception("Contabo settings not found. Please configure them first.");
        }
    }
    
    // Generate a valid UUID v4
    private function generateUuid() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
    
    // Get access token with authentication
    private function getAccessToken() {
        if (!$this->access_token) {
            $this->authenticate();
        }
        return $this->access_token;
    }
    
    // Make API request helper method
    private function makeRequest($method, $url, $data = null) {
        $accessToken = $this->getAccessToken();
        
        $ch = curl_init();
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $accessToken,
                'Content-Type: application/json',
                'x-request-id: ' . $this->generateUuid(),
                'x-trace-id: ' . $this->generateUuid()
            ]
        ];
        
        if ($method === 'POST') {
            $options[CURLOPT_POST] = true;
            if ($data) {
                $options[CURLOPT_POSTFIELDS] = json_encode($data);
            }
        } elseif ($method === 'PUT' || $method === 'PATCH' || $method === 'DELETE') {
            $options[CURLOPT_CUSTOMREQUEST] = $method;
            if ($data) {
                $options[CURLOPT_POSTFIELDS] = json_encode($data);
            }
        }
        
        curl_setopt_array($ch, $options);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        // Log the response for debugging
        error_log("Contabo API $method $url - HTTP: $http_code, Response: " . $response);
        
        if ($http_code < 200 || $http_code >= 300) {
            $error_msg = "API request failed: HTTP $http_code";
            if ($curl_error) {
                $error_msg .= " - cURL Error: " . $curl_error;
            }
            if ($response) {
                $decoded_response = json_decode($response, true);
                if (isset($decoded_response['message'])) {
                    $error_msg .= " - Message: " . $decoded_response['message'];
                } else {
                    $error_msg .= " - Response: " . $response;
                }
            }
            throw new Exception($error_msg);
        }
        
        return json_decode($response, true);
    }
    
    public function authenticate() {
        $auth_data = [
            'client_id' => $this->settings['client_id'],
            'client_secret' => $this->settings['client_secret'],
            'username' => $this->settings['api_user'],
            'password' => $this->settings['api_password'],
            'grant_type' => 'password'
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::CONTABO_AUTH_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($auth_data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded'
            ]
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            throw new Exception("Authentication failed: HTTP $http_code - " . $response);
        }
        
        $data = json_decode($response, true);
        $this->access_token = $data['access_token'];
        
        return $this->access_token;
    }
    
    // Add cancel instance method
    public function cancelInstance($instanceId, $cancelDate = null) {
        // Default to 30 days from now if no date provided
        if (!$cancelDate) {
            $cancelDate = date('Y-m-d', strtotime('+30 days'));
        }
        
        $data = [
            'cancelDate' => $cancelDate
        ];
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/cancel', $data);
    }
    
    // Add remove cancellation method
    public function removeCancellation($instanceId) {
        // Note: Contabo API might not have a direct "remove cancellation" endpoint
        // This would typically be done by updating the instance to remove cancellation
        // For now, we'll throw an informative error
        throw new Exception("Remove cancellation functionality is not directly available in the Contabo API. Please contact Contabo support to remove cancellation.");
    }
    
    // Add rollback snapshot method
    public function rollbackSnapshot($instanceId, $snapshotId) {
        $data = new stdClass(); // Empty object as required by API
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots/' . $snapshotId . '/actions/rollback', $data);
    }
    
    public function getInstances() {
        return $this->makeRequest('GET', Config::CONTABO_API_BASE . '/compute/instances');
    }
    
    public function getInstanceDetails($instanceId) {
        return $this->makeRequest('GET', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId);
    }
    
    public function getAvailableImages() {
        return $this->makeRequest('GET', Config::CONTABO_API_BASE . '/compute/images');
    }
    
    // Get popular OS images with better filtering
    public function getPopularOSImages() {
        $images_data = $this->getAvailableImages();
        $popular_images = [];
        
        if (isset($images_data['data']) && is_array($images_data['data'])) {
            foreach ($images_data['data'] as $image) {
                $name = strtolower($image['name'] ?? '');
                $osType = strtolower($image['osType'] ?? '');
                
                // Filter for popular distributions
                if (strpos($name, 'ubuntu') !== false || 
                    strpos($name, 'debian') !== false ||
                    strpos($name, 'centos') !== false ||
                    strpos($name, 'alma') !== false || // AlmaLinux
                    strpos($name, 'rocky') !== false ||
                    strpos($name, 'fedora') !== false ||
                    strpos($name, 'windows') !== false ||
                    strpos($name, 'freebsd') !== false) {
                    $popular_images[] = $image;
                }
            }
        }
        
        return $popular_images;
    }
    
    public function reinstallInstance($instanceId, $imageId, $sshKeys = [], $rootPassword = null, $userData = '', $defaultUser = 'admin') {
        $data = [
            'imageId' => $imageId,
            'defaultUser' => $defaultUser
        ];
        
        if (!empty($sshKeys)) {
            $data['sshKeys'] = $sshKeys;
        }
        
        if ($rootPassword) {
            $data['rootPassword'] = $rootPassword;
        }
        
        if (!empty($userData)) {
            $data['userData'] = $userData;
        }
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/reinstall', $data);
    }
    
    public function createSnapshot($instanceId, $name, $description = '') {
        $data = [
            'name' => $name,
            'description' => $description
        ];
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots', $data);
    }
    
    public function getSnapshots($instanceId) {
        return $this->makeRequest('GET', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots');
    }
    
    public function deleteSnapshot($instanceId, $snapshotId) {
        return $this->makeRequest('DELETE', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/snapshots/' . $snapshotId);
    }
    
    public function resetPassword($instanceId, $sshKeys = [], $rootPassword = null, $userData = '') {
        $data = [];
        
        if (!empty($sshKeys)) {
            $data['sshKeys'] = $sshKeys;
        }
        
        if ($rootPassword) {
            $data['rootPassword'] = $rootPassword;
        }
        
        if (!empty($userData)) {
            $data['userData'] = $userData;
        }
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/resetPassword', $data);
    }
    
    public function rescueInstance($instanceId, $rootPassword = null, $sshKeys = [], $userData = '') {
        $data = [];
        
        if ($rootPassword) {
            $data['rootPassword'] = $rootPassword;
        }
        
        if (!empty($sshKeys)) {
            $data['sshKeys'] = $sshKeys;
        }
        
        if (!empty($userData)) {
            $data['userData'] = $userData;
        }
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/rescue', $data);
    }
    
    public function startInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'start');
    }
    
    public function stopInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'stop');
    }
    
    public function restartInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'restart');
    }
    
    public function shutdownInstance($instanceId) {
        return $this->performInstanceAction($instanceId, 'shutdown');
    }
    
    private function performInstanceAction($instanceId, $action) {
        // FIXED: Send empty JSON object as required by Contabo API
        $data = new stdClass();
        
        return $this->makeRequest('POST', Config::CONTABO_API_BASE . '/compute/instances/' . $instanceId . '/actions/' . $action, $data);
    }
    
    public function syncInstancesToDB() {
        try {
            $instances_data = $this->getInstances();
            $db = getDB();
            
            // Clear existing instances
            $db->query("TRUNCATE TABLE vps_instances");
            
            $stmt = $db->prepare("
                INSERT INTO vps_instances 
                (instance_id, name, status, ip_address, created_date, region, image_id, product_type) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            foreach ($instances_data['data'] as $instance) {
                $ip_address = '';
                if (isset($instance['ipConfig']['v4']['ip']) && !empty($instance['ipConfig']['v4']['ip'])) {
                    $ip_address = $instance['ipConfig']['v4']['ip'][0];
                }
                
                $stmt->execute([
                    $instance['instanceId'],
                    $instance['name'],
                    $instance['status'],
                    $ip_address,
                    date('Y-m-d H:i:s', strtotime($instance['createdDate'])),
                    $instance['region'],
                    $instance['imageId'],
                    $instance['productType']
                ]);
            }
            
            return count($instances_data['data']);
            
        } catch(Exception $e) {
            throw new Exception("Sync failed: " . $e->getMessage());
        }
    }
    
    // FIXED: Auto-sync instances without creating duplicates
    public function autoSyncInstances() {
        try {
            $instances_data = $this->getInstances();
            $db = getDB();
            
            // First, get all existing instance IDs to track what should be removed
            $existingStmt = $db->query("SELECT instance_id FROM vps_instances");
            $existing_instances = $existingStmt->fetchAll(PDO::FETCH_COLUMN);
            $current_instance_ids = [];
            
            // Prepare statements for insert and update
            $insertStmt = $db->prepare("
                INSERT INTO vps_instances 
                (instance_id, name, status, ip_address, created_date, region, image_id, product_type) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $updateStmt = $db->prepare("
                UPDATE vps_instances 
                SET name = ?, status = ?, ip_address = ?, created_date = ?, region = ?, image_id = ?, product_type = ?
                WHERE instance_id = ?
            ");
            
            $count = 0;
            foreach ($instances_data['data'] as $instance) {
                $ip_address = '';
                if (isset($instance['ipConfig']['v4']['ip']) && !empty($instance['ipConfig']['v4']['ip'])) {
                    $ip_address = $instance['ipConfig']['v4']['ip'][0];
                }
                
                $instance_id = $instance['instanceId'];
                $current_instance_ids[] = $instance_id;
                
                // Check if instance already exists
                $checkStmt = $db->prepare("SELECT 1 FROM vps_instances WHERE instance_id = ?");
                $checkStmt->execute([$instance_id]);
                
                if ($checkStmt->fetch()) {
                    // Update existing instance
                    $updateStmt->execute([
                        $instance['name'],
                        $instance['status'],
                        $ip_address,
                        date('Y-m-d H:i:s', strtotime($instance['createdDate'])),
                        $instance['region'],
                        $instance['imageId'],
                        $instance['productType'],
                        $instance_id
                    ]);
                } else {
                    // Insert new instance
                    $insertStmt->execute([
                        $instance_id,
                        $instance['name'],
                        $instance['status'],
                        $ip_address,
                        date('Y-m-d H:i:s', strtotime($instance['createdDate'])),
                        $instance['region'],
                        $instance['imageId'],
                        $instance['productType']
                    ]);
                }
                $count++;
            }
            
            // Remove instances that no longer exist in Contabo (optional - be careful with this)
            // $instances_to_remove = array_diff($existing_instances, $current_instance_ids);
            // if (!empty($instances_to_remove)) {
            //     $placeholders = str_repeat('?,', count($instances_to_remove) - 1) . '?';
            //     $deleteStmt = $db->prepare("DELETE FROM vps_instances WHERE instance_id IN ($placeholders)");
            //     $deleteStmt->execute(array_values($instances_to_remove));
            // }
            
            return $count;
            
        } catch(Exception $e) {
            throw new Exception("Auto-sync failed: " . $e->getMessage());
        }
    }
    
    // Log actions to database
    public function logAction($instanceId, $actionType, $actionData = '') {
        try {
            $db = getDB();
            $stmt = $db->prepare("
                INSERT INTO instance_actions_log 
                (instance_id, action_type, action_data, performed_by) 
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([
                $instanceId,
                $actionType,
                $actionData,
                $_SESSION['user_id']
            ]);
            return true;
        } catch(Exception $e) {
            // Don't throw error if logging fails
            error_log("Failed to log action: " . $e->getMessage());
            return false;
        }
    }
 
    /**
     * Create a new VPS instance
     */
    public function createInstance($data) {
        $url = Config::CONTABO_API_BASE . '/compute/instances';
        
        $payload = [
            'imageId' => $data['imageId'] ?? 'afecbb85-e2fc-46f0-9684-b46b1faf00bb', // Ubuntu 22.04 default
            'productId' => $data['productId'],
            'region' => $data['region'] ?? 'EU',
            'period' => $data['period'] ?? 1,
            'displayName' => $data['displayName'] ?? 'VPS Instance',
            'defaultUser' => $data['defaultUser'] ?? 'admin'
        ];
        
        // Add root password if provided
        if (!empty($data['rootPassword'])) {
            $payload['rootPassword'] = $data['rootPassword'];
        }
        
        // Add cloud-init config if provided (for clearing MOTD)
        if (!empty($data['userData'])) {
            $payload['userData'] = $data['userData'];
        }
        
        // Optional parameters - SSH Keys removed since we don't use them in the form
        /*
        if (!empty($data['sshKeys'])) {
            $payload['sshKeys'] = $data['sshKeys'];
        }
        */
        
        if (!empty($data['license'])) {
            $payload['license'] = $data['license'];
        }
        
        $response = $this->makeRequest('POST', $url, $payload);
        
        // Log the creation
        $this->logVpsAction('new', 'create', $data);
        
        return $response;
    }
    
    /**
     * Log VPS creation action
     */
    private function logVpsAction($instanceId, $action, $data = []) {
        try {
            $db = getDB();
            $stmt = $db->prepare("
                INSERT INTO vps_actions (instance_id, action, action_data, performed_by, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $instanceId,
                $action,
                json_encode($data),
                $_SESSION['user_id'] ?? null
            ]);
        } catch (Exception $e) {
            error_log("Log VPS Action Error: " . $e->getMessage());
        }
    }
    
    /**
     * Save packages to database or file
     */
    public static function savePackages($packages) {
        try {
            $db = getDB();
            
            // Clear existing packages
            $db->query("DELETE FROM vps_packages");
            
            // Insert updated packages
            $stmt = $db->prepare("
                INSERT INTO vps_packages 
                (package_id, name, cpu, ram, disk, disk_type, snapshots, traffic, backup, price_monthly, price_original, currency, description)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            foreach ($packages as $packageId => $package) {
                $stmt->execute([
                    $packageId,
                    $package['name'],
                    $package['cpu'],
                    $package['ram'],
                    $package['disk'],
                    $package['disk_type'],
                    $package['snapshots'],
                    $package['traffic'],
                    $package['backup'] ? 1 : 0,
                    $package['price_monthly'],
                    $package['price_original'],
                    $package['currency'],
                    $package['description']
                ]);
            }
            
            return true;
        } catch (Exception $e) {
            error_log("Save packages error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Load packages from database
     */
    public static function loadPackages() {
        try {
            $db = getDB();
            $stmt = $db->query("SELECT * FROM vps_packages");
            $dbPackages = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $packages = [];
            foreach ($dbPackages as $dbPackage) {
                $packages[$dbPackage['package_id']] = [
                    'name' => $dbPackage['name'],
                    'cpu' => (int)$dbPackage['cpu'],
                    'ram' => (int)$dbPackage['ram'],
                    'disk' => (int)$dbPackage['disk'],
                    'disk_type' => $dbPackage['disk_type'],
                    'snapshots' => (int)$dbPackage['snapshots'],
                    'traffic' => $dbPackage['traffic'],
                    'backup' => (bool)$dbPackage['backup'],
                    'price_monthly' => (float)$dbPackage['price_monthly'],
                    'price_original' => (float)$dbPackage['price_original'],
                    'currency' => $dbPackage['currency'],
                    'description' => $dbPackage['description']
                ];
            }
            
            return !empty($packages) ? $packages : self::getDefaultVpsPackages();
        } catch (Exception $e) {
            error_log("Load packages error: " . $e->getMessage());
            return self::getDefaultVpsPackages();
        }
    }
    
    /**
     * Get VPS packages configuration - now loads from database
     */
    public static function getVpsPackages() {
        return self::loadPackages();
    }
    
    /**
     * Get default VPS packages (fallback)
     */
    private static function getDefaultVpsPackages() {
        return [
            'V91' => [
                'name' => 'Cloud VPS 10',
                'cpu' => 3,
                'ram' => 8,
                'disk' => 75,
                'disk_type' => 'NVMe',
                'snapshots' => 1,
                'traffic' => '32 TB',
                'backup' => false,
                'price_monthly' => 4.02,
                'price_original' => 5.36,
                'currency' => '€',
                'description' => 'Unbeatable Value'
            ],
            'V92' => [
                'name' => 'Cloud VPS 10 SSD',
                'cpu' => 3,
                'ram' => 8,
                'disk' => 150,
                'disk_type' => 'SSD',
                'snapshots' => 1,
                'traffic' => '32 TB',
                'backup' => false,
                'price_monthly' => 4.02,
                'price_original' => 5.36,
                'currency' => '€',
                'description' => 'Unbeatable Value'
            ],
            'V94' => [
                'name' => 'Cloud VPS 20',
                'cpu' => 6,
                'ram' => 12,
                'disk' => 100,
                'disk_type' => 'NVMe',
                'snapshots' => 2,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 6.25,
                'price_original' => 8.33,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V95' => [
                'name' => 'Cloud VPS 20 SSD',
                'cpu' => 6,
                'ram' => 12,
                'disk' => 200,
                'disk_type' => 'SSD',
                'snapshots' => 2,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 6.25,
                'price_original' => 8.33,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V97' => [
                'name' => 'Cloud VPS 30',
                'cpu' => 8,
                'ram' => 24,
                'disk' => 200,
                'disk_type' => 'NVMe',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 12.50,
                'price_original' => 16.66,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V98' => [
                'name' => 'Cloud VPS 30 SSD',
                'cpu' => 8,
                'ram' => 24,
                'disk' => 400,
                'disk_type' => 'SSD',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 12.50,
                'price_original' => 16.66,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V100' => [
                'name' => 'Cloud VPS 40',
                'cpu' => 12,
                'ram' => 48,
                'disk' => 250,
                'disk_type' => 'NVMe',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 22.31,
                'price_original' => 29.75,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V101' => [
                'name' => 'Cloud VPS 40 SSD',
                'cpu' => 12,
                'ram' => 48,
                'disk' => 500,
                'disk_type' => 'SSD',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 22.31,
                'price_original' => 29.75,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V103' => [
                'name' => 'Cloud VPS 50',
                'cpu' => 16,
                'ram' => 64,
                'disk' => 300,
                'disk_type' => 'NVMe',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 33.02,
                'price_original' => 44.03,
                'currency' => 'MK',
                'description' => 'Free Auto Backup'
            ],
            'V104' => [
                'name' => 'Cloud VPS 50 SSD',
                'cpu' => 16,
                'ram' => 64,
                'disk' => 600,
                'disk_type' => 'SSD',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 33.02,
                'price_original' => 44.03,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V106' => [
                'name' => 'Cloud VPS 60',
                'cpu' => 18,
                'ram' => 96,
                'disk' => 350,
                'disk_type' => 'NVMe',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 43.73,
                'price_original' => 58.31,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ],
            'V107' => [
                'name' => 'Cloud VPS 60 SSD',
                'cpu' => 18,
                'ram' => 96,
                'disk' => 700,
                'disk_type' => 'SSD',
                'snapshots' => 3,
                'traffic' => '32 TB',
                'backup' => true,
                'price_monthly' => 43.73,
                'price_original' => 58.31,
                'currency' => '€',
                'description' => 'Free Auto Backup'
            ]
        ];
    }
    
    /**
     * Get available regions
     */
    public static function getRegions() {
        return [
            'EU' => 'European Union',
            'US-central' => 'US Central',
            'US-east' => 'US East', 
            'US-west' => 'US West',
            'SIN' => 'Singapore',
            'UK' => 'United Kingdom',
            'AUS' => 'Australia',
            'JPN' => 'Japan',
            'IND' => 'India'
        ];
    }
    
    /**
     * Get contract periods
     */
    public static function getPeriods() {
        return [
            1 => '1 Month',
            3 => '3 Months',
            6 => '6 Months', 
            12 => '12 Months'
        ];
    }
    
    // Helper function to fix IP address display
    public function formatIPAddresses($ipConfig) {
        $formatted = [
            'ipv4' => [],
            'ipv6' => []
        ];
        
        if (isset($ipConfig['v4']['ip'])) {
            if (is_array($ipConfig['v4']['ip'])) {
                $formatted['ipv4'] = $ipConfig['v4']['ip'];
            } elseif (is_string($ipConfig['v4']['ip'])) {
                $formatted['ipv4'] = [$ipConfig['v4']['ip']];
            }
        }
        
        if (isset($ipConfig['v6']['ip'])) {
            if (is_array($ipConfig['v6']['ip'])) {
                $formatted['ipv6'] = $ipConfig['v6']['ip'];
            } elseif (is_string($ipConfig['v6']['ip'])) {
                $formatted['ipv6'] = [$ipConfig['v6']['ip']];
            }
        }
        
        return $formatted;
    }
}
?>