<?php
// payment_logs.php
require_once 'config.php';
require_once 'contabo_api.php';

redirectIfNotLoggedIn();

// Check if user is admin
if (!isAdmin()) {
    header("Location: dashboard.php");
    exit;
}

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 50;
$offset = ($page - 1) * $limit;

// Search and filter
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$event_type = isset($_GET['event_type']) ? trim($_GET['event_type']) : '';
$date_from = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
$date_to = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';

// Build query
$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(transaction_id LIKE ? OR phone_number LIKE ? OR event_details LIKE ?)";
    $search_term = "%$search%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

if (!empty($event_type)) {
    $where_conditions[] = "event_type = ?";
    $params[] = $event_type;
}

if (!empty($date_from)) {
    $where_conditions[] = "DATE(created_at) >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "DATE(created_at) <= ?";
    $params[] = $date_to;
}

$where_sql = '';
if (!empty($where_conditions)) {
    $where_sql = 'WHERE ' . implode(' AND ', $where_conditions);
}

// Get total count
$db = getDB();
$count_stmt = $db->prepare("SELECT COUNT(*) FROM payment_logs $where_sql");
if (!empty($params)) {
    $count_stmt->execute($params);
} else {
    $count_stmt->execute();
}
$total_logs = $count_stmt->fetchColumn();

// Get logs
$logs_stmt = $db->prepare("
    SELECT pl.*, u.username, u.email 
    FROM payment_logs pl 
    LEFT JOIN users u ON pl.user_id = u.id 
    $where_sql 
    ORDER BY pl.created_at DESC 
    LIMIT $limit OFFSET $offset
");

if (!empty($params)) {
    $logs_stmt->execute($params);
} else {
    $logs_stmt->execute();
}
$logs = $logs_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get unique event types for filter
$event_types_stmt = $db->query("SELECT DISTINCT event_type FROM payment_logs ORDER BY event_type");
$event_types = $event_types_stmt->fetchAll(PDO::FETCH_COLUMN);

// Calculate pagination
$total_pages = ceil($total_logs / $limit);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Logs - Admin</title>
    <style>
        :root {
            --primary: #007bff;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --secondary: #6c757d;
            --light: #f8f9fa;
            --dark: #343a40;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .filters {
            background: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .filter-row {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr 1fr auto;
            gap: 15px;
            align-items: end;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        
        .filter-group label {
            margin-bottom: 5px;
            font-weight: 500;
            color: var(--dark);
        }
        
        input, select, button {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            font-weight: 500;
            transition: all 0.3s ease;
            text-align: center;
            display: inline-block;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-secondary {
            background: var(--secondary);
            color: white;
        }
        
        .btn-info {
            background: var(--info);
            color: white;
        }
        
        .logs-table {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        th {
            background: var(--light);
            font-weight: 600;
            color: var(--dark);
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .event-type {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .event-initiated { background: #e3f2fd; color: #1976d2; }
        .event-completed { background: #e8f5e8; color: #2e7d32; }
        .event-failed { background: #ffebee; color: #c62828; }
        .event-error { background: #fff3e0; color: #ef6c00; }
        .event-processing { background: #f3e5f5; color: #7b1fa2; }
        
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 10px;
        }
        
        .pagination-info {
            margin: 0 15px;
            color: var(--secondary);
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .stat-number {
            font-size: 2rem;
            font-weight: bold;
            color: var(--primary);
        }
        
        .stat-label {
            color: var(--secondary);
            margin-top: 5px;
        }
        
        .details-popup {
            max-width: 500px;
            word-wrap: break-word;
        }
        
        @media (max-width: 768px) {
            .filter-row {
                grid-template-columns: 1fr;
            }
            
            .stats {
                grid-template-columns: 1fr;
            }
            
            .logs-table {
                overflow-x: auto;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Payment Logs</h1>
            <p>Comprehensive payment transaction monitoring</p>
        </div>
        
        <div class="stats">
            <div class="stat-card">
                <div class="stat-number"><?php echo number_format($total_logs); ?></div>
                <div class="stat-label">Total Logs</div>
            </div>
            <?php
            // Get some quick stats
            $stats_stmt = $db->query("
                SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN event_type = 'COMPLETED' THEN 1 ELSE 0 END) as completed,
                    SUM(CASE WHEN event_type = 'FAILED' THEN 1 ELSE 0 END) as failed,
                    SUM(CASE WHEN event_type = 'ERROR' THEN 1 ELSE 0 END) as errors
                FROM payment_logs
            ");
            $stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
            ?>
            <div class="stat-card">
                <div class="stat-number"><?php echo number_format($stats['completed'] ?? 0); ?></div>
                <div class="stat-label">Completed</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo number_format($stats['failed'] ?? 0); ?></div>
                <div class="stat-label">Failed</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo number_format($stats['errors'] ?? 0); ?></div>
                <div class="stat-label">Errors</div>
            </div>
        </div>
        
        <div class="filters">
            <form method="GET" id="filterForm">
                <div class="filter-row">
                    <div class="filter-group">
                        <label for="search">Search</label>
                        <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Transaction ID, Phone, Details...">
                    </div>
                    
                    <div class="filter-group">
                        <label for="event_type">Event Type</label>
                        <select id="event_type" name="event_type">
                            <option value="">All Events</option>
                            <?php foreach ($event_types as $type): ?>
                                <option value="<?php echo $type; ?>" <?php echo $event_type === $type ? 'selected' : ''; ?>>
                                    <?php echo $type; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label for="date_from">Date From</label>
                        <input type="date" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                    </div>
                    
                    <div class="filter-group">
                        <label for="date_to">Date To</label>
                        <input type="date" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                    </div>
                    
                    <div class="filter-group">
                        <button type="submit" class="btn btn-primary">Apply Filters</button>
                        <a href="payment_logs.php" class="btn btn-secondary" style="margin-top: 5px;">Clear</a>
                    </div>
                </div>
            </form>
        </div>
        
        <div class="logs-table">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Transaction ID</th>
                        <th>User</th>
                        <th>Event Type</th>
                        <th>Phone</th>
                        <th>Amount</th>
                        <th>IP Address</th>
                        <th>Timestamp</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($logs)): ?>
                        <tr>
                            <td colspan="9" style="text-align: center; padding: 40px;">
                                No payment logs found.
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td><?php echo $log['id']; ?></td>
                                <td>
                                    <code><?php echo htmlspecialchars($log['transaction_id']); ?></code>
                                </td>
                                <td>
                                    <?php if ($log['username']): ?>
                                        <?php echo htmlspecialchars($log['username']); ?><br>
                                        <small style="color: var(--secondary);"><?php echo htmlspecialchars($log['email']); ?></small>
                                    <?php else: ?>
                                        <em>Guest</em>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="event-type event-<?php echo strtolower($log['event_type']); ?>">
                                        <?php echo htmlspecialchars($log['event_type']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($log['phone_number']); ?></td>
                                <td>
                                    <?php if ($log['amount'] > 0): ?>
                                        MWK <?php echo number_format($log['amount'], 2); ?>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <code><?php echo htmlspecialchars($log['ip_address']); ?></code>
                                </td>
                                <td>
                                    <?php echo date('M j, Y H:i:s', strtotime($log['created_at'])); ?>
                                </td>
                                <td>
                                    <button class="btn btn-info btn-sm" 
                                            onclick="showDetails('<?php echo addslashes($log['event_details']); ?>')">
                                        View Details
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        
        <?php if ($total_pages > 1): ?>
            <div class="pagination">
                <?php if ($page > 1): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" class="btn btn-primary">
                        Previous
                    </a>
                <?php endif; ?>
                
                <span class="pagination-info">
                    Page <?php echo $page; ?> of <?php echo $total_pages; ?>
                </span>
                
                <?php if ($page < $total_pages): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" class="btn btn-primary">
                        Next
                    </a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <script>
        function showDetails(details) {
            alert('Event Details:\n\n' + details);
        }
        
        // Auto-submit form when filters change
        document.getElementById('event_type').addEventListener('change', function() {
            document.getElementById('filterForm').submit();
        });
        
        document.getElementById('date_from').addEventListener('change', function() {
            document.getElementById('filterForm').submit();
        });
        
        document.getElementById('date_to').addEventListener('change', function() {
            document.getElementById('filterForm').submit();
        });
    </script>
</body>
</html>