<?php
// user_vps.php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';
require_once 'contabo_api.php'; // ADD THIS LINE - Fix the missing class

redirectIfNotLoggedIn();

// Different redirect based on role
if (isCustomer()) {
    header("Location: customer_vps.php");
    exit;
}

$message = '';
$error = '';
$user_instances = [];
$customers = [];

try {
    $db = getDB();
    $contabo = new ContaboAPI();
    
    // Auto-sync instances on page load
    try {
        $contabo->autoSyncInstances();
    } catch(Exception $e) {
        error_log("Auto-sync warning: " . $e->getMessage());
        // Continue even if sync fails
    }
    
    // Debug: Check if user_id exists in session
    if (!isset($_SESSION['user_id']) || empty($_SESSION['user_id'])) {
        throw new Exception("User not properly logged in");
    }
    
    // Get VPS instances assigned to the current user - FIXED QUERY TO PREVENT DUPLICATES
    $stmt = $db->prepare("
        SELECT DISTINCT vi.*, uva.customer_id, c.username as customer_username
        FROM vps_instances vi 
        INNER JOIN user_vps_assignments uva ON vi.instance_id = uva.instance_id 
        LEFT JOIN customers c ON uva.customer_id = c.id
        WHERE uva.user_id = ? 
        ORDER BY vi.created_date DESC
    ");
    
    if (!$stmt) {
        throw new Exception("Failed to prepare statement: " . implode(", ", $db->errorInfo()));
    }
    
    $stmt->execute([$_SESSION['user_id']]);
    $user_instances = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Debug output
    error_log("Found " . count($user_instances) . " instances for user " . $_SESSION['user_id']);
    
    // Get customers for this store
    $customerStmt = $db->prepare("SELECT id, username, email FROM customers WHERE store_user_id = ? AND status = 'active' ORDER BY username");
    
    if (!$customerStmt) {
        throw new Exception("Failed to prepare customer statement: " . implode(", ", $db->errorInfo()));
    }
    
    $customerStmt->execute([$_SESSION['user_id']]);
    $customers = $customerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Handle instance actions
    if (isset($_GET['action']) && isset($_GET['instance_id'])) {
        $action = $_GET['action'];
        $instanceId = $_GET['instance_id'];
        
        // Verify user has access to this instance
        $checkStmt = $db->prepare("
            SELECT 1 FROM user_vps_assignments 
            WHERE user_id = ? AND instance_id = ?
        ");
        
        if (!$checkStmt) {
            throw new Exception("Failed to prepare check statement");
        }
        
        $checkStmt->execute([$_SESSION['user_id'], $instanceId]);
        
        if ($checkStmt->fetch()) {
            $contabo = new ContaboAPI();
            
            switch($action) {
                case 'start':
                    $contabo->startInstance($instanceId);
                    $contabo->logAction($instanceId, 'start');
                    $message = "Instance started successfully!";
                    break;
                case 'stop':
                    $contabo->stopInstance($instanceId);
                    $contabo->logAction($instanceId, 'stop');
                    $message = "Instance stopped successfully!";
                    break;
                case 'restart':
                    $contabo->restartInstance($instanceId);
                    $contabo->logAction($instanceId, 'restart');
                    $message = "Instance restarted successfully!";
                    break;
                    
                case 'assign_customer':
                    if (isset($_GET['customer_id']) && !empty($_GET['customer_id'])) {
                        $customerId = $_GET['customer_id'];
                        $updateStmt = $db->prepare("UPDATE user_vps_assignments SET customer_id = ? WHERE instance_id = ? AND user_id = ?");
                        if ($updateStmt->execute([$customerId, $instanceId, $_SESSION['user_id']])) {
                            $message = "VPS assigned to customer successfully!";
                        } else {
                            $error = "Failed to assign VPS to customer";
                        }
                    }
                    break;
                    
                case 'unassign_customer':
                    $updateStmt = $db->prepare("UPDATE user_vps_assignments SET customer_id = NULL WHERE instance_id = ? AND user_id = ?");
                    if ($updateStmt->execute([$instanceId, $_SESSION['user_id']])) {
                        $message = "VPS unassigned from customer successfully!";
                    } else {
                        $error = "Failed to unassign VPS from customer";
                    }
                    break;
                    
                default:
                    $error = "Unknown action";
                    break;
            }
            
            // Redirect to avoid form resubmission
            header("Location: user_vps.php?message=" . urlencode($message) . "&error=" . urlencode($error));
            exit;
            
        } else {
            $error = "You don't have permission to perform this action on this instance.";
        }
    }
    
    // Check for message/error from redirect
    if (isset($_GET['message'])) {
        $message = $_GET['message'];
    }
    if (isset($_GET['error'])) {
        $error = $_GET['error'];
    }
    
} catch(Exception $e) {
    $error = "Error: " . $e->getMessage();
    error_log("User VPS Error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My VPS - <?php echo htmlspecialchars($_SESSION['username'] ?? 'User'); ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background-color: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; position: relative; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .message.fade-out {
            opacity: 0;
            transition: opacity 0.5s ease-in-out;
        }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .status-running { color: #28a745; font-weight: bold; }
        .status-stopped { color: #dc3545; font-weight: bold; }
        .status-provisioning { color: #ffc107; font-weight: bold; }
        .btn { display: inline-block; padding: 6px 12px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .btn-primary { background: #007bff; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; flex-wrap: wrap; }
        .actions { display: flex; gap: 5px; flex-wrap: wrap; }
        .customer-assignment { margin: 5px 0; }
        .nav-links { margin-bottom: 20px; padding-bottom: 10px; border-bottom: 1px solid #ddd; }
        .nav-links a { margin-right: 15px; text-decoration: none; color: #007bff; }
        .nav-links a:hover { text-decoration: underline; }
        .no-data { text-align: center; padding: 40px; color: #6c757d; }
        select { padding: 6px; border: 1px solid #ddd; border-radius: 4px; }
        .refresh-info { background: #e7f3ff; padding: 10px; border-radius: 4px; margin-bottom: 15px; font-size: 14px; }
    </style>
    <script>
        // Auto-hide messages after 5 seconds
        document.addEventListener('DOMContentLoaded', function() {
            const messages = document.querySelectorAll('.message');
            messages.forEach(function(message) {
                setTimeout(function() {
                    message.classList.add('fade-out');
                    setTimeout(function() {
                        if (message.parentNode) {
                            message.parentNode.removeChild(message);
                        }
                    }, 500);
                }, 5000);
            });
        });
    </script>
</head>
<body>
    <div class="container">
        <div class="nav-links">
            <a href="user_vps.php">My VPS</a>
            <a href="user_customers.php">My Customers</a>
            <a href="user_store_settings.php">Store Settings</a>
            <a href="logout.php">Logout</a>
        </div>

        <div class="header">
            <h1>My VPS Instances</h1>
            <div>
                <span>Welcome, <?php echo htmlspecialchars($_SESSION['username'] ?? 'User'); ?> (Store Owner)</span>
                <?php if (isAdmin()): ?>
                    <a href="view_vps.php" class="btn btn-info">Admin View</a>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="refresh-info">
            <strong>Auto-Refresh:</strong> Instance data is automatically synchronized with Contabo API on page load.
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if (empty($user_instances)): ?>
            <div class="no-data">
                <h3>No VPS Instances Found</h3>
                <p>No VPS instances are currently assigned to your store.</p>
                <p>Please contact the administrator to assign VPS instances to your store.</p>
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Instance ID</th>
                        <th>Status</th>
                        <th>IP Address</th>
                        <th>Assigned To Customer</th>
                        <th>Created Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($user_instances as $instance): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($instance['name'] ?? 'N/A'); ?></td>
                        <td><?php echo htmlspecialchars($instance['instance_id'] ?? 'N/A'); ?></td>
                        <td class="status-<?php echo strtolower($instance['status'] ?? 'unknown'); ?>">
                            <?php echo htmlspecialchars($instance['status'] ?? 'Unknown'); ?>
                        </td>
                        <td><?php echo htmlspecialchars($instance['ip_address'] ?? 'N/A'); ?></td>
                        <td>
                            <?php if (!empty($instance['customer_username'])): ?>
                                <div>
                                    <strong><?php echo htmlspecialchars($instance['customer_username']); ?></strong>
                                    <br>
                                    <a href="user_vps.php?action=unassign_customer&instance_id=<?php echo $instance['instance_id']; ?>" 
                                       class="btn btn-danger btn-sm" 
                                       onclick="return confirm('Are you sure you want to unassign this VPS from the customer?')">
                                        Unassign
                                    </a>
                                </div>
                            <?php else: ?>
                                <div class="customer-assignment">
                                    <span style="color: #dc3545;">Not assigned</span>
                                    <?php if (!empty($customers)): ?>
                                        <div style="margin-top: 5px;">
                                            <form method="GET" style="display: inline;">
                                                <input type="hidden" name="action" value="assign_customer">
                                                <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                                <select name="customer_id" onchange="this.form.submit()" style="padding: 4px;">
                                                    <option value="">Assign to customer...</option>
                                                    <?php foreach ($customers as $customer): ?>
                                                        <option value="<?php echo $customer['id']; ?>">
                                                            <?php echo htmlspecialchars($customer['username'] . ' (' . $customer['email'] . ')'); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </form>
                                        </div>
                                    <?php else: ?>
                                        <div style="margin-top: 5px;">
                                            <small style="color: #6c757d;">No active customers available</small>
                                            <br>
                                            <a href="user_customers.php" class="btn btn-primary btn-sm">Add Customers</a>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                        </td>
                        <td><?php echo !empty($instance['created_date']) ? date('M j, Y H:i', strtotime($instance['created_date'])) : 'N/A'; ?></td>
                        <td class="actions">
                            <a href="user_vps.php?action=start&instance_id=<?php echo $instance['instance_id']; ?>" 
                               class="btn btn-success" 
                               onclick="return confirm('Are you sure you want to start this instance?')">Start</a>
                            <a href="user_vps.php?action=stop&instance_id=<?php echo $instance['instance_id']; ?>" 
                               class="btn btn-danger" 
                               onclick="return confirm('Are you sure you want to stop this instance?')">Stop</a>
                            <a href="user_vps.php?action=restart&instance_id=<?php echo $instance['instance_id']; ?>" 
                               class="btn btn-warning" 
                               onclick="return confirm('Are you sure you want to restart this instance?')">Restart</a>
                            <a href="instance_details.php?instance_id=<?php echo $instance['instance_id']; ?>" 
                               class="btn btn-info">Manage</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</body>
</html>