<?php
// view_vps.php
require_once 'config.php';
require_once 'contabo_api.php'; // Ensure this is included

redirectIfNotLoggedIn();
redirectIfNotAdmin(); // Only admins can access this page

$message = '';
$error = '';
$instances = [];
$users = [];
$customers = [];
$search_term = '';

try {
    $contabo = new ContaboAPI();
    
    // Handle manual sync
    if (isset($_GET['sync']) && $_GET['sync'] === 'true') {
        try {
            $count = $contabo->syncInstancesToDB();
            $message = "Successfully synced $count instances from Contabo API!";
        } catch(Exception $e) {
            $error = "Manual sync failed: " . $e->getMessage();
            error_log("Manual sync error: " . $e->getMessage());
        }
    } else {
        // Auto-sync instances on page load (only if not manually syncing)
        try {
            $contabo->autoSyncInstances();
        } catch(Exception $e) {
            error_log("Auto-sync warning: " . $e->getMessage());
            // Continue even if sync fails
        }
    }
    
    // Handle search
    if (isset($_GET['search']) && !empty($_GET['search'])) {
        $search_term = trim($_GET['search']);
    }
    
    // Handle instance actions
    if (isset($_GET['action']) && isset($_GET['instance_id'])) {
        $action = $_GET['action'];
        $instanceId = $_GET['instance_id'];
        
        switch($action) {
            case 'start':
                $contabo->startInstance($instanceId);
                $contabo->logAction($instanceId, 'start');
                $message = "Instance started successfully!";
                break;
            case 'stop':
                $contabo->stopInstance($instanceId);
                $contabo->logAction($instanceId, 'stop');
                $message = "Instance stopped successfully!";
                break;
            case 'restart':
                $contabo->restartInstance($instanceId);
                $contabo->logAction($instanceId, 'restart');
                $message = "Instance restarted successfully!";
                break;
            case 'assign_user':
                if (isset($_GET['user_id'])) {
                    $userId = $_GET['user_id'];
                    $db = getDB();
                    
                    // Check if assignment already exists
                    $checkStmt = $db->prepare("SELECT id FROM user_vps_assignments WHERE user_id = ? AND instance_id = ?");
                    $checkStmt->execute([$userId, $instanceId]);
                    
                    if (!$checkStmt->fetch()) {
                        $stmt = $db->prepare("INSERT INTO user_vps_assignments (user_id, instance_id) VALUES (?, ?)");
                        $stmt->execute([$userId, $instanceId]);
                        
                        // Also update the vps_instances table
                        $updateStmt = $db->prepare("UPDATE vps_instances SET assigned_to = ? WHERE instance_id = ?");
                        $updateStmt->execute([$userId, $instanceId]);
                        
                        $message = "VPS assigned to user successfully!";
                    } else {
                        $message = "VPS is already assigned to this user!";
                    }
                }
                break;
            case 'assign_customer':
                if (isset($_GET['customer_id'])) {
                    $customerId = $_GET['customer_id'];
                    $db = getDB();
                    
                    // Check if instance is assigned to a user first
                    $checkUserStmt = $db->prepare("SELECT user_id FROM user_vps_assignments WHERE instance_id = ?");
                    $checkUserStmt->execute([$instanceId]);
                    $userAssignment = $checkUserStmt->fetch();
                    
                    if ($userAssignment) {
                        // Update existing assignment with customer
                        $updateStmt = $db->prepare("UPDATE user_vps_assignments SET customer_id = ? WHERE instance_id = ?");
                        $updateStmt->execute([$customerId, $instanceId]);
                        $message = "VPS assigned to customer successfully!";
                    } else {
                        $message = "Error: VPS must be assigned to a user first before assigning to customer!";
                    }
                }
                break;
            case 'unassign_user':
                $db = getDB();
                $stmt = $db->prepare("DELETE FROM user_vps_assignments WHERE instance_id = ?");
                $stmt->execute([$instanceId]);
                
                // Also update the vps_instances table
                $updateStmt = $db->prepare("UPDATE vps_instances SET assigned_to = NULL WHERE instance_id = ?");
                $updateStmt->execute([$instanceId]);
                
                $message = "VPS unassigned from user successfully!";
                break;
            case 'unassign_customer':
                $db = getDB();
                $updateStmt = $db->prepare("UPDATE user_vps_assignments SET customer_id = NULL WHERE instance_id = ?");
                $updateStmt->execute([$instanceId]);
                $message = "VPS unassigned from customer successfully!";
                break;
        }
        
        // Redirect to avoid form resubmission
        $redirect_url = "view_vps.php?message=" . urlencode($message) . "&error=" . urlencode($error);
        if (!empty($search_term)) {
            $redirect_url .= "&search=" . urlencode($search_term);
        }
        header("Location: " . $redirect_url);
        exit;
    }
    
    // Load instances from database with user and customer info
    $db = getDB();
    $query = "
        SELECT DISTINCT 
            vi.*, 
            u.username as assigned_username,
            u.id as user_id,
            c.username as customer_username,
            c.id as customer_id,
            s.username as store_username
        FROM vps_instances vi 
        LEFT JOIN user_vps_assignments uva ON vi.instance_id = uva.instance_id
        LEFT JOIN users u ON uva.user_id = u.id 
        LEFT JOIN customers c ON uva.customer_id = c.id
        LEFT JOIN users s ON uva.user_id = s.id
    ";
    
    // Add search condition if search term exists
    if (!empty($search_term)) {
        $query .= " WHERE vi.name LIKE :search OR vi.instance_id LIKE :search";
    }
    
    $query .= " ORDER BY vi.created_date DESC";
    
    $stmt = $db->prepare($query);
    
    if (!empty($search_term)) {
        $search_param = "%" . $search_term . "%";
        $stmt->bindValue(':search', $search_param);
    }
    
    $stmt->execute();
    $instances = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Load users for assignment
    $userStmt = $db->query("SELECT id, username FROM users WHERE role = 'user' ORDER BY username");
    $users = $userStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Load customers for assignment
    $customerStmt = $db->query("SELECT id, username, store_user_id FROM customers WHERE status = 'active' ORDER BY username");
    $customers = $customerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Check for message/error from redirect
    if (isset($_GET['message'])) {
        $message = $_GET['message'];
    }
    if (isset($_GET['error'])) {
        $error = $_GET['error'];
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
    error_log("Admin VPS Error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage VPS Instances - Admin</title>
    <style>
        :root {
            --primary: #007bff;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --secondary: #6c757d;
            --light: #f8f9fa;
            --dark: #343a40;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            margin: 0; 
            background-color: #f5f5f5; 
            line-height: 1.6;
        }
        
        .container { 
            max-width: 1400px; 
            margin: 0 auto; 
            background: white; 
            padding: 20px; 
            border-radius: 12px; 
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            min-height: 100vh;
        }
        
        .message { 
            padding: 12px 16px; 
            margin: 16px 0; 
            border-radius: 8px; 
            position: relative;
            font-weight: 500;
        }
        
        .success { 
            background: #d4edda; 
            color: #155724; 
            border: 1px solid #c3e6cb; 
        }
        
        .error { 
            background: #f8d7da; 
            color: #721c24; 
            border: 1px solid #f5c6cb; 
        }
        
        .message.fade-out {
            opacity: 0;
            transition: opacity 0.5s ease-in-out;
        }
        
        /* Table Styles for Desktop */
        .table-container {
            overflow-x: auto;
            margin-top: 20px;
        }
        
        table { 
            width: 100%; 
            border-collapse: collapse; 
            margin-top: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        th, td { 
            padding: 14px; 
            text-align: left; 
            border-bottom: 1px solid #e9ecef; 
        }
        
        th { 
            background-color: #f8f9fa; 
            font-weight: 600;
            color: var(--dark);
            border-top: 1px solid #dee2e6;
        }
        
        .status-running { color: var(--success); font-weight: 600; }
        .status-stopped { color: var(--danger); font-weight: 600; }
        .status-provisioning { color: var(--warning); font-weight: 600; }
        
        .btn { 
            display: inline-block; 
            padding: 8px 16px; 
            margin: 4px; 
            border: none; 
            border-radius: 6px; 
            cursor: pointer; 
            text-decoration: none; 
            font-size: 13px; 
            font-weight: 500;
            transition: all 0.2s ease;
            text-align: center;
        }
        
        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }
        
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }
        
        .btn-success { background: var(--success); color: white; }
        .btn-warning { background: var(--warning); color: black; }
        .btn-danger { background: var(--danger); color: white; }
        .btn-info { background: var(--info); color: white; }
        .btn-primary { background: var(--primary); color: white; }
        .btn-secondary { background: var(--secondary); color: white; }
        .btn-light { background: var(--light); color: var(--dark); }
        
        .header { 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            margin-bottom: 24px; 
            flex-wrap: wrap;
            gap: 16px;
        }
        
        .actions { 
            display: flex; 
            gap: 6px; 
            flex-wrap: wrap; 
        }
        
        .assignment-section { 
            margin: 12px 0; 
            padding: 12px; 
            border: 1px solid #e9ecef; 
            border-radius: 6px; 
        }
        
        .assignment-form { 
            display: inline-block; 
            margin-left: 12px; 
        }
        
        .user-assignment { 
            background: #e7f3ff; 
        }
        
        .customer-assignment { 
            background: #f0f9ff; 
        }
        
        .assignment-label { 
            font-weight: 600; 
            margin-bottom: 6px; 
            color: var(--dark);
        }
        
        select { 
            padding: 8px; 
            border: 1px solid #ddd; 
            border-radius: 6px; 
            background: white;
            font-size: 14px;
        }
        
        .nav-links { 
            margin-bottom: 24px; 
            padding-bottom: 16px; 
            border-bottom: 1px solid #e9ecef; 
            display: flex;
            flex-wrap: wrap;
            gap: 12px;
        }
        
        .nav-links a { 
            text-decoration: none; 
            color: var(--primary); 
            font-weight: 500;
            padding: 8px 12px;
            border-radius: 6px;
            transition: background-color 0.2s;
        }
        
        .nav-links a:hover { 
            background-color: #f8f9fa;
            text-decoration: none;
        }
        
        .assignment-badges { 
            display: flex; 
            flex-direction: column; 
            gap: 8px; 
        }
        
        .badge { 
            display: inline-block; 
            padding: 6px 12px; 
            border-radius: 6px; 
            font-size: 12px; 
            font-weight: 600; 
        }
        
        .badge-user { 
            background: var(--success); 
            color: white; 
        }
        
        .badge-customer { 
            background: var(--info); 
            color: white; 
        }
        
        .refresh-info { 
            background: #e7f3ff; 
            padding: 16px; 
            border-radius: 8px; 
            margin-bottom: 20px; 
            font-size: 14px;
            border-left: 4px solid var(--primary);
        }
        
        .refresh-controls { 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            margin-bottom: 20px; 
            flex-wrap: wrap;
            gap: 16px;
        }
        
        .manual-refresh { 
            display: flex; 
            align-items: center; 
            gap: 12px; 
            flex-wrap: wrap;
        }
        
        .last-sync { 
            color: var(--secondary); 
            font-size: 14px; 
        }
        
        /* Search Styles */
        .search-container {
            margin-bottom: 20px;
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
            align-items: center;
        }
        
        .search-box {
            flex: 1;
            min-width: 250px;
            position: relative;
        }
        
        .search-input {
            width: 100%;
            padding: 12px 16px;
            padding-right: 40px;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.2s;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0,123,255,0.1);
        }
        
        .search-btn {
            position: absolute;
            right: 8px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--secondary);
            cursor: pointer;
        }
        
        .clear-search {
            background: var(--light);
            color: var(--dark);
        }
        
        /* Searchable Dropdown Styles */
        .searchable-dropdown {
            position: relative;
            display: inline-block;
            min-width: 200px;
        }
        
        .dropdown-search {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            margin-bottom: 5px;
        }
        
        .dropdown-search:focus {
            outline: none;
            border-color: var(--primary);
        }
        
        .dropdown-list {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #ddd;
            border-radius: 6px;
            max-height: 200px;
            overflow-y: auto;
            z-index: 1000;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            display: none;
        }
        
        .dropdown-list.show {
            display: block;
        }
        
        .dropdown-option {
            padding: 10px 12px;
            cursor: pointer;
            border-bottom: 1px solid #f5f5f5;
            transition: background-color 0.2s;
        }
        
        .dropdown-option:hover {
            background-color: #f8f9fa;
        }
        
        .dropdown-option:last-child {
            border-bottom: none;
        }
        
        .dropdown-select {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            background: white;
            font-size: 14px;
            cursor: pointer;
        }
        
        /* Card Styles for Mobile */
        .cards-container {
            display: none;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .instance-card {
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        .instance-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.12);
        }
        
        .card-header {
            display: flex;
            justify-content: between;
            align-items: flex-start;
            margin-bottom: 16px;
            gap: 12px;
        }
        
        .card-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 4px;
        }
        
        .card-id {
            font-size: 12px;
            color: var(--secondary);
            font-family: monospace;
        }
        
        .card-status {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            white-space: nowrap;
        }
        
        .card-details {
            margin-bottom: 16px;
        }
        
        .detail-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #f8f9fa;
        }
        
        .detail-label {
            font-weight: 500;
            color: var(--secondary);
        }
        
        .detail-value {
            color: var(--dark);
            text-align: right;
        }
        
        .card-assignments {
            margin: 16px 0;
            padding: 16px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        
        .assignment-item {
            margin-bottom: 12px;
        }
        
        .assignment-item:last-child {
            margin-bottom: 0;
        }
        
        .card-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-top: 16px;
        }
        
        .card-actions .btn {
            flex: 1;
            min-width: 80px;
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .container {
                padding: 16px;
                border-radius: 0;
            }
            
            .header {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .nav-links {
                flex-direction: column;
                gap: 8px;
            }
            
            .nav-links a {
                padding: 12px;
                background: #f8f9fa;
            }
            
            .refresh-controls {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .search-container {
                flex-direction: column;
            }
            
            .search-box {
                min-width: 100%;
            }
            
            table {
                display: none;
            }
            
            .cards-container {
                display: grid;
            }
            
            .cards-container {
                grid-template-columns: 1fr;
            }
            
            .btn {
                padding: 10px 14px;
                font-size: 14px;
            }
            
            .searchable-dropdown {
                min-width: 100%;
            }
        }
        
        @media (min-width: 769px) and (max-width: 1024px) {
            .cards-container {
                grid-template-columns: repeat(2, 1fr);
            }
            
            table {
                display: none;
            }
            
            .cards-container {
                display: grid;
            }
        }
        
        @media (min-width: 1025px) {
            .cards-container {
                display: none;
            }
            
            table {
                display: table;
            }
        }
        
        .instance-count {
            background: var(--light);
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 14px;
            color: var(--dark);
            margin-bottom: 16px;
        }
        
        .no-instances {
            text-align: center;
            padding: 40px 20px;
            color: var(--secondary);
        }
        
        .no-instances i {
            font-size: 48px;
            margin-bottom: 16px;
            opacity: 0.5;
        }
        
        .dropdown-header {
            padding: 8px 12px;
            background: #f8f9fa;
            font-weight: 600;
            color: var(--dark);
            border-bottom: 1px solid #e9ecef;
        }
        
        .no-results {
            padding: 12px;
            text-align: center;
            color: var(--secondary);
            font-style: italic;
        }
    </style>
    <script>
        // Auto-hide messages after 5 seconds
        document.addEventListener('DOMContentLoaded', function() {
            const messages = document.querySelectorAll('.message');
            messages.forEach(function(message) {
                setTimeout(function() {
                    message.classList.add('fade-out');
                    setTimeout(function() {
                        if (message.parentNode) {
                            message.parentNode.removeChild(message);
                        }
                    }, 500);
                }, 5000);
            });
            
            // Add loading state to manual refresh button
            const refreshBtn = document.getElementById('manualRefreshBtn');
            if (refreshBtn) {
                refreshBtn.addEventListener('click', function() {
                    this.innerHTML = '<span>Refreshing...</span>';
                    this.disabled = true;
                });
            }
            
            // Clear search functionality
            const clearSearchBtn = document.getElementById('clearSearch');
            if (clearSearchBtn) {
                clearSearchBtn.addEventListener('click', function() {
                    window.location.href = 'view_vps.php';
                });
            }
            
            // Enter key search
            const searchInput = document.getElementById('searchInput');
            if (searchInput) {
                searchInput.addEventListener('keypress', function(e) {
                    if (e.key === 'Enter') {
                        this.form.submit();
                    }
                });
            }
            
            // Initialize searchable dropdowns
            initSearchableDropdowns();
        });
        
        // Searchable dropdown functionality
        function initSearchableDropdowns() {
            const dropdowns = document.querySelectorAll('.searchable-dropdown');
            
            dropdowns.forEach(dropdown => {
                const select = dropdown.querySelector('.dropdown-select');
                const search = dropdown.querySelector('.dropdown-search');
                const list = dropdown.querySelector('.dropdown-list');
                const hiddenInput = dropdown.querySelector('input[type="hidden"]');
                const options = list.querySelectorAll('.dropdown-option');
                
                // Show dropdown when clicking select
                select.addEventListener('click', function() {
                    list.classList.toggle('show');
                    if (list.classList.contains('show')) {
                        search.focus();
                    }
                });
                
                // Filter options based on search
                search.addEventListener('input', function() {
                    const searchTerm = this.value.toLowerCase();
                    let hasVisibleOptions = false;
                    
                    options.forEach(option => {
                        const text = option.textContent.toLowerCase();
                        if (text.includes(searchTerm)) {
                            option.style.display = 'block';
                            hasVisibleOptions = true;
                        } else {
                            option.style.display = 'none';
                        }
                    });
                    
                    // Show/hide no results message
                    const noResults = list.querySelector('.no-results');
                    if (noResults) {
                        noResults.style.display = hasVisibleOptions ? 'none' : 'block';
                    }
                });
                
                // Select option
                options.forEach(option => {
                    option.addEventListener('click', function() {
                        const value = this.getAttribute('data-value');
                        const text = this.textContent;
                        
                        select.textContent = text;
                        if (hiddenInput) {
                            hiddenInput.value = value;
                        }
                        
                        list.classList.remove('show');
                        search.value = '';
                        
                        // Reset all options to visible
                        options.forEach(opt => opt.style.display = 'block');
                        
                        // Submit the form if it's an auto-submit dropdown
                        const form = dropdown.closest('form');
                        if (form && dropdown.classList.contains('auto-submit')) {
                            form.submit();
                        }
                    });
                });
                
                // Close dropdown when clicking outside
                document.addEventListener('click', function(e) {
                    if (!dropdown.contains(e.target)) {
                        list.classList.remove('show');
                    }
                });
            });
        }
        
        // Function to create searchable dropdown HTML
        function createSearchableDropdown(name, options, placeholder, autoSubmit = true) {
            const dropdownId = name + '_dropdown';
            return `
                <div class="searchable-dropdown ${autoSubmit ? 'auto-submit' : ''}">
                    <div class="dropdown-select">${placeholder}</div>
                    <div class="dropdown-list">
                        <input type="text" class="dropdown-search" placeholder="Search ${placeholder.toLowerCase()}...">
                        <div class="dropdown-options">
                            ${options.map(option => 
                                `<div class="dropdown-option" data-value="${option.value}">${option.text}</div>`
                            ).join('')}
                            <div class="no-results" style="display: none;">No results found</div>
                        </div>
                    </div>
                    <input type="hidden" name="${name}" value="">
                </div>
            `;
        }
    </script>
</head>
<body>
    <div class="container">
        <div class="nav-links">
            <a href="view_vps.php">VPS Management</a>
            <a href="user_management.php">User Management</a>
            <a href="customer_management.php">Customer Management</a>
            <a href="settings.php">Settings</a>
            <a href="logout.php">Logout</a>
        </div>

        <div class="header">
            <h1>Manage VPS Instances (Admin)</h1>
            <div>
                <span>Welcome, <?php echo htmlspecialchars($_SESSION['username']); ?> (Admin)</span>
                <a href="user_vps.php" class="btn btn-info">User View</a>
            </div>
        </div>
        
        <div class="refresh-controls">
            <div class="refresh-info">
                <strong>Auto-Refresh:</strong> Instance data is automatically synchronized with Contabo API on page load. 
                You can also manually refresh using the button.
            </div>
            <div class="manual-refresh">
                <span class="last-sync">Last sync: <?php echo date('M j, Y H:i:s'); ?></span>
                <a href="view_vps.php?sync=true<?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-success" id="manualRefreshBtn">
                    <span>Manual Refresh</span>
                </a>
            </div>
        </div>
        
        <!-- Search Box -->
        <div class="search-container">
            <form method="GET" class="search-box">
                <input type="text" 
                       name="search" 
                       id="searchInput"
                       class="search-input" 
                       placeholder="Search by name or instance ID..." 
                       value="<?php echo htmlspecialchars($search_term); ?>">
                <button type="submit" class="search-btn">🔍</button>
            </form>
            <?php if (!empty($search_term)): ?>
                <a href="view_vps.php" class="btn btn-secondary clear-search" id="clearSearch">
                    Clear Search
                </a>
            <?php endif; ?>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <?php if (empty($instances)): ?>
            <div class="no-instances">
                <div>📋</div>
                <h3>No VPS Instances Found</h3>
                <p><?php echo !empty($search_term) ? 'No instances match your search criteria.' : 'Instances will appear automatically once they are available in your Contabo account.'; ?></p>
                <?php if (!empty($search_term)): ?>
                    <a href="view_vps.php" class="btn btn-primary">View All Instances</a>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <!-- Instance Count -->
            <div class="instance-count">
                Showing <?php echo count($instances); ?> instance(s)
                <?php if (!empty($search_term)): ?>
                    for "<?php echo htmlspecialchars($search_term); ?>"
                <?php endif; ?>
            </div>
            
            <!-- Desktop Table View -->
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Instance ID</th>
                            <th>Status</th>
                            <th>IP Address</th>
                            <th>Region</th>
                            <th>Assignments</th>
                            <th>Created Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($instances as $instance): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($instance['name']); ?></td>
                            <td><?php echo htmlspecialchars($instance['instance_id']); ?></td>
                            <td class="status-<?php echo strtolower($instance['status']); ?>">
                                <?php echo htmlspecialchars($instance['status']); ?>
                            </td>
                            <td><?php echo htmlspecialchars($instance['ip_address']); ?></td>
                            <td><?php echo htmlspecialchars($instance['region']); ?></td>
                            <td>
                                <div class="assignment-badges">
                                    <!-- User Assignment -->
                                    <?php if ($instance['assigned_username']): ?>
                                        <div>
                                            <span class="badge badge-user">Store Owner: <?php echo htmlspecialchars($instance['assigned_username']); ?></span>
                                            <a href="view_vps.php?action=unassign_user&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" 
                                               class="btn btn-danger btn-sm" 
                                               onclick="return confirm('Unassign this VPS from store owner?')">Unassign</a>
                                        </div>
                                    <?php else: ?>
                                        <div class="assignment-section user-assignment">
                                            <div class="assignment-label">Assign to Store Owner:</div>
                                            <div class="assignment-form">
                                                <form method="GET" style="display: inline;">
                                                    <input type="hidden" name="action" value="assign_user">
                                                    <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                                    <?php if (!empty($search_term)): ?>
                                                        <input type="hidden" name="search" value="<?php echo htmlspecialchars($search_term); ?>">
                                                    <?php endif; ?>
                                                    <div class="searchable-dropdown auto-submit">
                                                        <div class="dropdown-select">Select store owner...</div>
                                                        <div class="dropdown-list">
                                                            <input type="text" class="dropdown-search" placeholder="Search store owners...">
                                                            <div class="dropdown-options">
                                                                <?php foreach ($users as $user): ?>
                                                                    <div class="dropdown-option" data-value="<?php echo $user['id']; ?>">
                                                                        <?php echo htmlspecialchars($user['username']); ?>
                                                                    </div>
                                                                <?php endforeach; ?>
                                                                <div class="no-results" style="display: none;">No store owners found</div>
                                                            </div>
                                                        </div>
                                                        <input type="hidden" name="user_id" value="">
                                                    </div>
                                                </form>
                                            </div>
                                        </div>
                                    <?php endif; ?>

                                    <!-- Customer Assignment -->
                                    <?php if ($instance['customer_username']): ?>
                                        <div>
                                            <span class="badge badge-customer">Customer: <?php echo htmlspecialchars($instance['customer_username']); ?></span>
                                            <a href="view_vps.php?action=unassign_customer&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" 
                                               class="btn btn-danger btn-sm" 
                                               onclick="return confirm('Unassign this VPS from customer?')">Unassign</a>
                                        </div>
                                    <?php else: ?>
                                        <?php if ($instance['assigned_username']): ?>
                                            <div class="assignment-section customer-assignment">
                                                <div class="assignment-label">Assign to Customer:</div>
                                                <div class="assignment-form">
                                                    <form method="GET" style="display: inline;">
                                                        <input type="hidden" name="action" value="assign_customer">
                                                        <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                                        <?php if (!empty($search_term)): ?>
                                                            <input type="hidden" name="search" value="<?php echo htmlspecialchars($search_term); ?>">
                                                        <?php endif; ?>
                                                        <div class="searchable-dropdown auto-submit">
                                                            <div class="dropdown-select">Select customer...</div>
                                                            <div class="dropdown-list">
                                                                <input type="text" class="dropdown-search" placeholder="Search customers...">
                                                                <div class="dropdown-options">
                                                                    <?php foreach ($customers as $customer): ?>
                                                                        <div class="dropdown-option" data-value="<?php echo $customer['id']; ?>">
                                                                            <?php echo htmlspecialchars($customer['username']); ?>
                                                                        </div>
                                                                    <?php endforeach; ?>
                                                                    <div class="no-results" style="display: none;">No customers found</div>
                                                                </div>
                                                            </div>
                                                            <input type="hidden" name="customer_id" value="">
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <div style="color: #6c757d; font-size: 11px;">
                                                Assign to store owner first
                                            </div>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td><?php echo date('M j, Y H:i', strtotime($instance['created_date'])); ?></td>
                            <td class="actions">
                                <a href="view_vps.php?action=start&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-success" onclick="return confirm('Start this instance?')">Start</a>
                                <a href="view_vps.php?action=stop&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-danger" onclick="return confirm('Stop this instance?')">Stop</a>
                                <a href="view_vps.php?action=restart&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-warning" onclick="return confirm('Restart this instance?')">Restart</a>
                                <a href="instance_details.php?instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-info">Details</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Mobile Card View -->
            <div class="cards-container">
                <?php foreach ($instances as $instance): ?>
                <div class="instance-card">
                    <div class="card-header">
                        <div style="flex: 1;">
                            <div class="card-title"><?php echo htmlspecialchars($instance['name']); ?></div>
                            <div class="card-id"><?php echo htmlspecialchars($instance['instance_id']); ?></div>
                        </div>
                        <div class="card-status status-<?php echo strtolower($instance['status']); ?>">
                            <?php echo htmlspecialchars($instance['status']); ?>
                        </div>
                    </div>
                    
                    <div class="card-details">
                        <div class="detail-row">
                            <span class="detail-label">IP Address:</span>
                            <span class="detail-value"><?php echo htmlspecialchars($instance['ip_address']); ?></span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Region:</span>
                            <span class="detail-value"><?php echo htmlspecialchars($instance['region']); ?></span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Created:</span>
                            <span class="detail-value"><?php echo date('M j, Y H:i', strtotime($instance['created_date'])); ?></span>
                        </div>
                    </div>
                    
                    <div class="card-assignments">
                        <!-- User Assignment -->
                        <?php if ($instance['assigned_username']): ?>
                            <div class="assignment-item">
                                <div class="badge badge-user" style="margin-bottom: 8px;">Store Owner: <?php echo htmlspecialchars($instance['assigned_username']); ?></div>
                                <a href="view_vps.php?action=unassign_user&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" 
                                   class="btn btn-danger btn-sm" 
                                   onclick="return confirm('Unassign this VPS from store owner?')" style="width: 100%;">Unassign Store Owner</a>
                            </div>
                        <?php else: ?>
                            <div class="assignment-item">
                                <div class="assignment-label">Assign to Store Owner:</div>
                                <form method="GET" style="margin-top: 8px;">
                                    <input type="hidden" name="action" value="assign_user">
                                    <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                    <?php if (!empty($search_term)): ?>
                                        <input type="hidden" name="search" value="<?php echo htmlspecialchars($search_term); ?>">
                                    <?php endif; ?>
                                    <div class="searchable-dropdown auto-submit">
                                        <div class="dropdown-select">Select store owner...</div>
                                        <div class="dropdown-list">
                                            <input type="text" class="dropdown-search" placeholder="Search store owners...">
                                            <div class="dropdown-options">
                                                <?php foreach ($users as $user): ?>
                                                    <div class="dropdown-option" data-value="<?php echo $user['id']; ?>">
                                                        <?php echo htmlspecialchars($user['username']); ?>
                                                    </div>
                                                <?php endforeach; ?>
                                                <div class="no-results" style="display: none;">No store owners found</div>
                                            </div>
                                        </div>
                                        <input type="hidden" name="user_id" value="">
                                    </div>
                                </form>
                            </div>
                        <?php endif; ?>

                        <!-- Customer Assignment -->
                        <?php if ($instance['customer_username']): ?>
                            <div class="assignment-item">
                                <div class="badge badge-customer" style="margin-bottom: 8px;">Customer: <?php echo htmlspecialchars($instance['customer_username']); ?></div>
                                <a href="view_vps.php?action=unassign_customer&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" 
                                   class="btn btn-danger btn-sm" 
                                   onclick="return confirm('Unassign this VPS from customer?')" style="width: 100%;">Unassign Customer</a>
                            </div>
                        <?php else: ?>
                            <?php if ($instance['assigned_username']): ?>
                                <div class="assignment-item">
                                    <div class="assignment-label">Assign to Customer:</div>
                                    <form method="GET" style="margin-top: 8px;">
                                        <input type="hidden" name="action" value="assign_customer">
                                        <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                        <?php if (!empty($search_term)): ?>
                                            <input type="hidden" name="search" value="<?php echo htmlspecialchars($search_term); ?>">
                                        <?php endif; ?>
                                        <div class="searchable-dropdown auto-submit">
                                            <div class="dropdown-select">Select customer...</div>
                                            <div class="dropdown-list">
                                                <input type="text" class="dropdown-search" placeholder="Search customers...">
                                                <div class="dropdown-options">
                                                    <?php foreach ($customers as $customer): ?>
                                                        <div class="dropdown-option" data-value="<?php echo $customer['id']; ?>">
                                                            <?php echo htmlspecialchars($customer['username']); ?>
                                                        </div>
                                                    <?php endforeach; ?>
                                                    <div class="no-results" style="display: none;">No customers found</div>
                                                </div>
                                            </div>
                                            <input type="hidden" name="customer_id" value="">
                                        </div>
                                    </form>
                                </div>
                            <?php else: ?>
                                <div style="color: #6c757d; font-size: 12px; text-align: center; padding: 8px;">
                                    Assign to store owner first
                                </div>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                    
                    <div class="card-actions">
                        <a href="view_vps.php?action=start&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-success" onclick="return confirm('Start this instance?')">Start</a>
                        <a href="view_vps.php?action=stop&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-danger" onclick="return confirm('Stop this instance?')">Stop</a>
                        <a href="view_vps.php?action=restart&instance_id=<?php echo $instance['instance_id']; ?><?php echo !empty($search_term) ? '&search=' . urlencode($search_term) : ''; ?>" class="btn btn-warning" onclick="return confirm('Restart this instance?')">Restart</a>
                        <a href="instance_details.php?instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-info">Details</a>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <p><small>Total instances: <?php echo count($instances); ?></small></p>
        <?php endif; ?>
    </div>
</body>
</html>