<?php
// vps_configure.php
require_once 'config.php';
require_once 'contabo_api.php';

redirectIfNotLoggedIn();

$message = '';
$error = '';
$package = null;
$packageId = '';

// Get package from URL parameter
if (isset($_GET['package'])) {
    $packageId = $_GET['package'];
    $packages = ContaboAPI::getVpsPackages();
    $package = $packages[$packageId] ?? null;
}

if (!$package) {
    header("Location: vps_packages.php");
    exit;
}

// Get available configurations
$regions = ContaboAPI::getRegions();
$periods = ContaboAPI::getPeriods();
$contabo = new ContaboAPI();
$images = $contabo->getAvailableImages();

// Handle form submission - redirect to payment
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Validate passwords match
        if ($_POST['root_password'] !== $_POST['confirm_password']) {
            throw new Exception("Passwords do not match!");
        }
        
        // Validate password length
        if (strlen($_POST['root_password']) < 8) {
            throw new Exception("Password must be at least 8 characters long!");
        }
        
        // Store configuration in session for payment processing
        $_SESSION['vps_order'] = [
            'package_id' => $packageId,
            'package_data' => $package,
            'configuration' => [
                'region' => $_POST['region'],
                'period' => (int)$_POST['period'],
                'display_name' => $_POST['display_name'],
                'image_id' => $_POST['image_id'],
                'default_user' => $_POST['default_user'],
                'root_password' => $_POST['root_password'], // Store the password
                'user_data' => $_POST['user_data'] // This now contains our cloud-init config
            ]
        ];
        
        // Redirect to payment page
        header("Location: vps_payment.php");
        exit;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configure VPS - <?php echo htmlspecialchars($package['name']); ?></title>
    <style>
        :root {
            --primary: #007bff;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --secondary: #6c757d;
            --light: #f8f9fa;
            --dark: #343a40;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 30px;
        }
        
        .header {
            grid-column: 1 / -1;
            text-align: center;
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .configuration-form {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        
        .package-summary {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            height: fit-content;
            position: sticky;
            top: 20px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: var(--dark);
        }
        
        input, select, textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus, textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0,123,255,0.1);
        }
        
        textarea {
            resize: vertical;
            min-height: 100px;
            font-family: monospace;
        }
        
        .btn {
            display: inline-block;
            padding: 12px 30px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            text-decoration: none;
            font-weight: 500;
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-success {
            background: var(--success);
            color: white;
        }
        
        .btn-block {
            width: 100%;
            display: block;
        }
        
        .btn-lg {
            padding: 15px 30px;
            font-size: 1.1rem;
        }
        
        .package-name {
            font-size: 1.3rem;
            font-weight: bold;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .package-features {
            list-style: none;
            margin-bottom: 20px;
        }
        
        .package-features li {
            padding: 8px 0;
            border-bottom: 1px solid #f0f0f0;
            display: flex;
            align-items: center;
        }
        
        .package-features li:last-child {
            border-bottom: none;
        }
        
        .feature-icon {
            color: var(--success);
            margin-right: 10px;
            font-weight: bold;
        }
        
        .package-price {
            text-align: center;
            padding: 20px;
            background: var(--light);
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .price-monthly {
            font-size: 2rem;
            font-weight: bold;
            color: var(--primary);
        }
        
        .price-original {
            font-size: 1.2rem;
            color: var(--secondary);
            text-decoration: line-through;
        }
        
        .message {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .form-help {
            font-size: 0.85rem;
            color: var(--secondary);
            margin-top: 5px;
        }
        
        .password-match {
            border-color: var(--success) !important;
            box-shadow: 0 0 0 3px rgba(40,167,69,0.1) !important;
        }
        
        .password-mismatch {
            border-color: var(--danger) !important;
            box-shadow: 0 0 0 3px rgba(220,53,69,0.1) !important;
        }
        
        @media (max-width: 768px) {
            .container {
                grid-template-columns: 1fr;
            }
            
            .package-summary {
                position: static;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Configure Your VPS</h1>
            <p>Customize your <?php echo htmlspecialchars($package['name']); ?> instance</p>
        </div>
        
        <div class="configuration-form">
            <?php if ($message): ?>
                <div class="message success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="message error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <form method="POST" id="vpsConfigForm">
                <div class="form-group">
                    <label for="display_name">Instance Name *</label>
                    <input type="text" id="display_name" name="display_name" 
                           value="My <?php echo htmlspecialchars($package['name']); ?>" 
                           required maxlength="255">
                    <div class="form-help">A descriptive name for your VPS instance</div>
                </div>
                
                <div class="form-group">
                    <label for="region">Region *</label>
                    <select id="region" name="region" required>
                        <?php foreach ($regions as $value => $name): ?>
                            <option value="<?php echo $value; ?>" 
                                    <?php echo $value === 'EU' ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="form-help">Choose the data center location for your VPS</div>
                </div>
                
                <div class="form-group">
                    <label for="period">Contract Period *</label>
                    <select id="period" name="period" required>
                        <?php foreach ($periods as $value => $name): ?>
                            <option value="<?php echo $value; ?>">
                                <?php echo htmlspecialchars($name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="image_id">Operating System *</label>
                    <select id="image_id" name="image_id" required>
                        <option value="afecbb85-e2fc-46f0-9684-b46b1faf00bb">Ubuntu 22.04 (Default)</option>
                        <option value="23b6c5c5-39c3-4f80-8fae-2b4fc65b5c5c">Ubuntu 20.04</option>
                        <option value="d65ecf3b-30db-4dc2-9e88-dfc21a14a6bc">Debian 11</option>
                        <option value="a1b2c3d4-e5f6-7890-abcd-ef1234567890">CentOS 9</option>
                        <option value="b2c3d4e5-f6g7-8901-bcde-f23456789012">Windows Server 2022</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="default_user">Default User *</label>
                    <select id="default_user" name="default_user" required>
                        <option value="admin">admin (Linux/BSD)</option>
                        <option value="root">root (Linux/BSD)</option>
                        <option value="administrator">administrator (Windows)</option>
                    </select>
                    <div class="form-help">Initial administrative user for the OS</div>
                </div>
                
                <div class="form-group">
                    <label for="root_password">Root/Admin Password *</label>
                    <input type="password" id="root_password" name="root_password" 
                           required minlength="8" placeholder="Enter root/administrator password">
                    <div class="form-help">Password for the default user (min 8 characters)</div>
                </div>

                <div class="form-group">
                    <label for="confirm_password">Confirm Password *</label>
                    <input type="password" id="confirm_password" name="confirm_password" 
                           required minlength="8" placeholder="Confirm the password">
                </div>
                
                <!-- Hidden field for cloud-init config to clear MOTD -->
                <input type="hidden" name="user_data" id="user_data" value="">
                
                <button type="submit" class="btn btn-success btn-block btn-lg">
                    Proceed to Payment
                </button>
            </form>
        </div>
        
        <div class="package-summary">
            <div class="package-name"><?php echo htmlspecialchars($package['name']); ?></div>
            
            <div class="package-price">
                <div class="price-monthly">
                    <?php echo $package['currency'] . $package['price_monthly']; ?>
                </div>
                <div class="price-original">
                    <?php echo $package['currency'] . $package['price_original']; ?> /month
                </div>
            </div>
            
            <ul class="package-features">
                <li><span class="feature-icon">✓</span> <?php echo $package['cpu']; ?> vCPU Cores</li>
                <li><span class="feature-icon">✓</span> <?php echo $package['ram']; ?> GB RAM</li>
                <li><span class="feature-icon">✓</span> <?php echo $package['disk']; ?> GB <?php echo $package['disk_type']; ?></li>
                <li><span class="feature-icon">✓</span> <?php echo $package['snapshots']; ?> Snapshot<?php echo $package['snapshots'] > 1 ? 's' : ''; ?></li>
                <li><span class="feature-icon">✓</span> <?php echo $package['traffic']; ?> Traffic</li>
                <li><span class="feature-icon">✓</span> Unlimited Incoming</li>
                <?php if ($package['backup']): ?>
                    <li><span class="feature-icon">✓</span> Free Auto Backup</li>
                <?php endif; ?>
            </ul>
            
            <div style="text-align: center; margin-top: 20px;">
                <a href="vps_packages.php" class="btn btn-primary">Choose Different Package</a>
            </div>
        </div>
    </div>

    <script>
        document.getElementById('vpsConfigForm').addEventListener('submit', function(e) {
            const password = document.getElementById('root_password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            // Validate password match
            if (password !== confirmPassword) {
                e.preventDefault();
                alert('Passwords do not match!');
                return;
            }
            
            // Validate password length
            if (password.length < 8) {
                e.preventDefault();
                alert('Password must be at least 8 characters long!');
                return;
            }
            
            // Set cloud-init config to clear MOTD
            const cloudInitConfig = `#cloud-config
runcmd:
  - echo "" > /etc/motd
  - echo "" > /etc/issue
  - echo "" > /etc/issue.net
  - sed -i 's/#PrintMotd yes/PrintMotd no/g' /etc/ssh/sshd_config
  - systemctl restart sshd || service ssh restart
`;
            
            document.getElementById('user_data').value = cloudInitConfig;
            
            const submitBtn = this.querySelector('button[type="submit"]');
            submitBtn.innerHTML = 'Processing...';
            submitBtn.disabled = true;
        });

        // Password confirmation validation
        document.getElementById('confirm_password').addEventListener('input', function() {
            const password = document.getElementById('root_password').value;
            const confirmPassword = this.value;
            
            if (confirmPassword && password !== confirmPassword) {
                this.classList.add('password-mismatch');
                this.classList.remove('password-match');
            } else if (confirmPassword && password === confirmPassword) {
                this.classList.add('password-match');
                this.classList.remove('password-mismatch');
            } else {
                this.classList.remove('password-match', 'password-mismatch');
            }
        });

        // Real-time password strength feedback
        document.getElementById('root_password').addEventListener('input', function() {
            const confirmPassword = document.getElementById('confirm_password').value;
            const password = this.value;
            
            if (confirmPassword) {
                if (password !== confirmPassword) {
                    document.getElementById('confirm_password').classList.add('password-mismatch');
                    document.getElementById('confirm_password').classList.remove('password-match');
                } else {
                    document.getElementById('confirm_password').classList.add('password-match');
                    document.getElementById('confirm_password').classList.remove('password-mismatch');
                }
            }
        });
        
        // Update OS options based on user selection
        document.getElementById('default_user').addEventListener('change', function() {
            const imageSelect = document.getElementById('image_id');
            const isWindows = this.value === 'administrator';
            
            // Enable/disable Windows options based on user selection
            for (let option of imageSelect.options) {
                if (option.textContent.includes('Windows')) {
                    option.disabled = !isWindows;
                    if (isWindows && option.value.includes('Windows')) {
                        imageSelect.value = option.value;
                    }
                } else if (isWindows) {
                    option.disabled = true;
                } else {
                    option.disabled = false;
                    if (option.value.includes('afecbb85')) {
                        imageSelect.value = option.value;
                    }
                }
            }
        });
    </script>
</body>
</html>