<?php
// vps_payment.php
require_once 'config.php';
require_once 'contabo_api.php';

redirectIfNotLoggedIn();

// Check if order data exists in session
if (!isset($_SESSION['vps_order'])) {
    header("Location: vps_packages.php");
    exit;
}

$order = $_SESSION['vps_order'];
$package = $order['package_data'];
$configuration = $order['configuration'];

$message = '';
$error = '';

// Handle payment form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $phone_number = $_POST['phone_number'];
        
        // Validate phone number
        if (empty($phone_number) || strlen($phone_number) < 9) {
            throw new Exception("Please enter a valid phone number");
        }
        
        // Calculate amount (monthly price * period)
        $amount = $package['price_monthly'] * $configuration['period'];
        
        // Step 1: Initiate payment
        $payment_result = initiatePayment($amount, $phone_number);
        
        if (!$payment_result['success']) {
            throw new Exception("Payment initiation failed: " . $payment_result['message']);
        }
        
        $transaction_id = $payment_result['transaction_id'];
        
        // Log payment initiation to database
        logPaymentEventToDB($transaction_id, 'INITIATED', "Payment initiated for {$phone_number}, Amount: {$amount}", $phone_number, $amount);
        
        // Step 2: Wait for user to enter PIN and system to process
        $message = "⌛ Payment initiated! Please check your phone and enter your PIN...<br>";
        $message .= "📱 <strong>Phone:</strong> " . $phone_number . "<br>";
        $message .= "💰 <strong>Amount:</strong> " . $package['currency'] . number_format($amount, 2) . "<br>";
        $message .= "🔗 <strong>Transaction ID:</strong> " . $transaction_id . "<br><br>";
        $message .= "⏳ <em>Waiting for payment confirmation (this may take 15-30 seconds)...</em>";
        
        // Store transaction info in session for status checking
        $_SESSION['payment_check'] = [
            'transaction_id' => $transaction_id,
            'start_time' => time(),
            'max_wait' => 60, // 60 seconds max
            'phone_number' => $phone_number,
            'amount' => $amount,
            'package' => $package,
            'configuration' => $configuration,
            'order' => $order
        ];
        
        // Redirect to avoid browser timeout
        header("Location: vps_payment.php?check_status=1");
        exit;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
        // Log error to database
        if (isset($transaction_id)) {
            logPaymentEventToDB($transaction_id, 'INITIATION_FAILED', $e->getMessage(), $phone_number ?? '', $amount ?? 0);
        }
    }
}

// Handle payment status checking
if (isset($_GET['check_status']) && isset($_SESSION['payment_check'])) {
    $payment_data = $_SESSION['payment_check'];
    $transaction_id = $payment_data['transaction_id'];
    $start_time = $payment_data['start_time'];
    $max_wait = $payment_data['max_wait'];
    
    try {
        // Check if we've exceeded the maximum wait time
        $elapsed_time = time() - $start_time;
        if ($elapsed_time > $max_wait) {
            logPaymentEventToDB($transaction_id, 'TIMEOUT', "Payment timeout after {$elapsed_time}s", $payment_data['phone_number'], $payment_data['amount']);
            throw new Exception("Payment timeout after {$max_wait} seconds. Please check your Airtel Money transaction history or contact support.");
        }
        
        // Check payment status
        $payment_status = checkPaymentStatus($transaction_id);
        
        if (!$payment_status['success']) {
            // Log status check failure
            logPaymentEventToDB($transaction_id, 'STATUS_CHECK_FAILED', $payment_status['message'], $payment_data['phone_number'], $payment_data['amount']);
            
            // If status check fails but we're still within time, continue waiting
            if ($elapsed_time < 45) { // Give more time for early failures
                header("Refresh: 5; URL=vps_payment.php?check_status=1");
                $message = "⌛ Payment processing... Please wait<br>";
                $message .= "⏱️ <strong>Elapsed Time:</strong> " . $elapsed_time . " seconds<br>";
                $message .= "🔗 <strong>Transaction ID:</strong> " . $transaction_id . "<br>";
                $message .= "⏳ <em>Checking payment status... (Attempting auto-refresh)</em>";
            } else {
                throw new Exception("Unable to verify payment status. Please contact support with Transaction ID: " . $transaction_id);
            }
        } else {
            $status = $payment_status['status'];
            
            // Handle different statuses
            if ($status === 'TS') {
                // ✅ Payment successful - Create VPS instance
                logPaymentEventToDB($transaction_id, 'SUCCESS', "Payment successful, creating VPS instance", $payment_data['phone_number'], $payment_data['amount']);
                
                $contabo = new ContaboAPI();
                
                $vps_data = [
                    'productId' => $payment_data['order']['package_id'],
                    'region' => $payment_data['configuration']['region'],
                    'period' => $payment_data['configuration']['period'],
                    'displayName' => $payment_data['configuration']['display_name'],
                    'imageId' => $payment_data['configuration']['image_id'],
                    'defaultUser' => $payment_data['configuration']['default_user']
                ];
                
                // Optional fields
                if (!empty($payment_data['configuration']['ssh_keys'])) {
                    $vps_data['sshKeys'] = array_map('trim', explode(',', $payment_data['configuration']['ssh_keys']));
                }
                
                if (!empty($payment_data['configuration']['user_data'])) {
                    $vps_data['userData'] = $payment_data['configuration']['user_data'];
                }
                
                // Create the VPS instance
                $vps_result = $contabo->createInstance($vps_data);
                
                // Store in database
                $db = getDB();
                
                // Begin transaction for data consistency
                $db->beginTransaction();
                
                try {
                    // Store VPS instance
                    $stmt = $db->prepare("
                        INSERT INTO vps_instances (instance_id, name, product_id, region, status, ip_address, created_date, assigned_to) 
                        VALUES (?, ?, ?, ?, 'provisioning', '', NOW(), ?)
                    ");
                    
                    $instance_id = $vps_result['data'][0]['instanceId'] ?? 'pending';
                    $stmt->execute([
                        $instance_id,
                        $payment_data['configuration']['display_name'],
                        $payment_data['order']['package_id'],
                        $payment_data['configuration']['region'],
                        $_SESSION['user_id']
                    ]);
                    
                    // Store payment record
                    $payment_stmt = $db->prepare("
                        INSERT INTO vps_payments 
                        (user_id, instance_id, transaction_id, amount, phone_number, status, ip_address, user_agent, created_at) 
                        VALUES (?, ?, ?, ?, ?, 'completed', ?, ?, NOW())
                    ");
                    $payment_stmt->execute([
                        $_SESSION['user_id'],
                        $instance_id,
                        $transaction_id,
                        $payment_data['amount'],
                        $payment_data['phone_number'],
                        getClientIP(),
                        $_SERVER['HTTP_USER_AGENT'] ?? ''
                    ]);
                    
                    // Commit transaction
                    $db->commit();
                    
                    // Log VPS creation success
                    logPaymentEventToDB($transaction_id, 'VPS_CREATED', "VPS instance created: {$instance_id}, Processing time: {$elapsed_time}s", $payment_data['phone_number'], $payment_data['amount']);
                    
                } catch (Exception $e) {
                    $db->rollBack();
                    logPaymentEventToDB($transaction_id, 'VPS_CREATION_FAILED', "Database error: " . $e->getMessage(), $payment_data['phone_number'], $payment_data['amount']);
                    throw new Exception("Database error: " . $e->getMessage());
                }
                
                // Clear sessions
                unset($_SESSION['vps_order'], $_SESSION['payment_check']);
                
                $message = "✅ Payment successful! Your VPS is being created.<br><br>";
                $message .= "💰 <strong>Amount Paid:</strong> " . $payment_data['package']['currency'] . number_format($payment_data['amount'], 2) . "<br>";
                $message .= "📱 <strong>Phone Number:</strong> . " . $payment_data['phone_number'] . "<br>";
                $message .= "🔗 <strong>Transaction ID:</strong> " . $transaction_id . "<br>";
                $message .= "📦 <strong>Instance ID:</strong> " . $instance_id . "<br>";
                $message .= "📊 <strong>Airtel Money ID:</strong> " . ($payment_status['airtel_money_id'] ?? 'N/A') . "<br>";
                $message .= "⏱️ <strong>Processing Time:</strong> " . $elapsed_time . " seconds<br><br>";
                $message .= "🚀 <strong>VPS Configuration:</strong><br>";
                $message .= "&nbsp;&nbsp;&nbsp;• Name: " . htmlspecialchars($payment_data['configuration']['display_name']) . "<br>";
                $message .= "&nbsp;&nbsp;&nbsp;• Package: " . htmlspecialchars($payment_data['package']['name']) . "<br>";
                $message .= "&nbsp;&nbsp;&nbsp;• Region: " . htmlspecialchars($payment_data['configuration']['region']) . "<br>";
                $message .= "&nbsp;&nbsp;&nbsp;• Period: " . $payment_data['configuration']['period'] . " month(s)<br>";
                $message .= "&nbsp;&nbsp;&nbsp;• OS: " . htmlspecialchars($payment_data['configuration']['image_id']) . "<br><br>";
                $message .= "📋 Your VPS will be ready in a few minutes. You can check the status in your dashboard.";
                
            } elseif ($status === 'FL' || $status === 'FA') {
                // ❌ Payment failed
                logPaymentEventToDB($transaction_id, 'FAILED', "Payment failed with status: {$status} - " . $payment_status['message'], $payment_data['phone_number'], $payment_data['amount']);
                throw new Exception("Payment failed. Status: " . $status . " - " . $payment_status['message']);
            } else {
                // ⏳ Still processing
                logPaymentEventToDB($transaction_id, 'PROCESSING', "Payment still processing. Status: {$status}, Elapsed: {$elapsed_time}s", $payment_data['phone_number'], $payment_data['amount']);
                
                if ($elapsed_time > 30) {
                    $message = "⌛ Payment still processing...<br>";
                    $message .= "⏱️ <strong>Elapsed Time:</strong> " . $elapsed_time . " seconds<br>";
                    $message .= "🔗 <strong>Transaction ID:</strong> " . $transaction_id . "<br>";
                    $message .= "📱 <strong>Phone:</strong> " . $payment_data['phone_number'] . "<br>";
                    $message .= "💡 <strong>Note:</strong> This is taking longer than usual. You can:<br>";
                    $message .= "&nbsp;&nbsp;&nbsp;• Check your Airtel Money transaction history<br>";
                    $message .= "&nbsp;&nbsp;&nbsp;• Wait a few more minutes<br>";
                    $message .= "&nbsp;&nbsp;&nbsp;• Contact support if issue persists<br>";
                    $message .= "<br>⏳ <em>Auto-refreshing in 10 seconds...</em>";
                    
                    header("Refresh: 10; URL=vps_payment.php?check_status=1");
                } else {
                    $message = "⌛ Payment processing... Please wait<br>";
                    $message .= "⏱️ <strong>Elapsed Time:</strong> " . $elapsed_time . " seconds<br>";
                    $message .= "🔗 <strong>Transaction ID:</strong> " . $transaction_id . "<br>";
                    $message .= "⏳ <em>Checking payment status... (Auto-refresh in 5 seconds)</em>";
                    
                    header("Refresh: 5; URL=vps_payment.php?check_status=1");
                }
            }
        }
        
    } catch (Exception $e) {
        $error = $e->getMessage();
        // Log error to database
        logPaymentEventToDB($transaction_id, 'ERROR', $e->getMessage(), $payment_data['phone_number'] ?? '', $payment_data['amount'] ?? 0);
        // Clear payment check session on error
        unset($_SESSION['payment_check']);
    }
}

/**
 * Initiate payment with Airtel Money
 */
function initiatePayment($amount, $phone_number) {
    $token = '7ccd870a1e52a2c64e930ae71c19400e3401823872d5066e5a1fa51889793236';
    $correspondent = 'AIRTEL_MWI';
    
    // Clean phone number - REMOVE LEADING ZERO as per working test file
    $number = ltrim($phone_number, '0');
    
    $url = 'https://api.perezproperty.co.za/api/deposits.php';
    
    $headers = [
        'Authorization: ' . $token,
        'Content-Type: ' . 'application/x-www-form-urlencoded'
    ];
    
    $post_data = http_build_query([
        'amount' => $amount, 
        'number' => $number,
        'token' => $token,
        'correspondent' => $correspondent
    ]);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $post_data,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 30
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200) {
        return [
            'success' => false,
            'message' => "HTTP $http_code - Payment initiation failed"
        ];
    }
    
    $data = json_decode($response, true);
    
    if (isset($data['status']['success']) && $data['status']['success'] === true) {
        return [
            'success' => true,
            'transaction_id' => $data['data']['transaction']['id'],
            'message' => $data['status']['message']
        ];
    } else {
        return [
            'success' => false,
            'message' => $data['status']['message'] ?? 'Payment initiation failed'
        ];
    }
}

/**
 * Check payment status
 */
function checkPaymentStatus($transaction_id) {
    $token = '7ccd870a1e52a2c64e930ae71c19400e3401823872d5066e5a1fa51889793236';
    
    $url = 'https://api.perezproperty.co.za/api/status.php';
    
    $headers = [
        'Authorization: ' . $token,
        'Content-Type: ' . 'application/x-www-form-urlencoded'
    ];
    
    $post_data = http_build_query([
        'depositId' => $transaction_id,
        'token' => $token
    ]);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $post_data,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 15 // Shorter timeout for status checks
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    if ($http_code !== 200) {
        return [
            'success' => false,
            'message' => "HTTP $http_code - " . ($curl_error ?: 'Status check failed')
        ];
    }
    
    $data = json_decode($response, true);
    
    if (isset($data['status']['success']) && $data['status']['success'] === true) {
        return [
            'success' => true,
            'status' => $data['data']['transaction']['status'] ?? 'UNKNOWN',
            'message' => $data['data']['transaction']['message'] ?? '',
            'airtel_money_id' => $data['data']['transaction']['airtel_money_id'] ?? ''
        ];
    } else {
        return [
            'success' => false,
            'message' => $data['status']['message'] ?? 'Status check failed'
        ];
    }
}

/**
 * Log payment events to database
 */
function logPaymentEventToDB($transaction_id, $event_type, $event_details = '', $phone_number = '', $amount = 0) {
    try {
        $db = getDB();
        $stmt = $db->prepare("
            INSERT INTO payment_logs 
            (transaction_id, user_id, event_type, event_details, phone_number, amount, ip_address, user_agent, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $user_id = $_SESSION['user_id'] ?? null;
        $stmt->execute([
            $transaction_id,
            $user_id,
            $event_type,
            $event_details,
            $phone_number,
            $amount,
            getClientIP(),
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
        return true;
    } catch (Exception $e) {
        // Fallback to file logging if database fails
        error_log("Payment log DB error: " . $e->getMessage());
        return false;
    }
}

/**
 * Get client IP address
 */
function getClientIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment - <?php echo htmlspecialchars($package['name']); ?></title>
    <style>
        :root {
            --primary: #007bff;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --secondary: #6c757d;
            --light: #f8f9fa;
            --dark: #343a40;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
        }
        
        .header {
            grid-column: 1 / -1;
            text-align: center;
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .payment-form {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        
        .order-summary {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            height: fit-content;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: var(--dark);
        }
        
        input, select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0,123,255,0.1);
        }
        
        .btn {
            display: inline-block;
            padding: 12px 30px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            text-decoration: none;
            font-weight: 500;
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .btn-success {
            background: var(--success);
            color: white;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-warning {
            background: var(--warning);
            color: black;
        }
        
        .btn-danger {
            background: var(--danger);
            color: white;
        }
        
        .btn-info {
            background: var(--info);
            color: white;
        }
        
        .btn-block {
            width: 100%;
            display: block;
        }
        
        .btn-lg {
            padding: 15px 30px;
            font-size: 1.1rem;
        }
        
        .package-name {
            font-size: 1.3rem;
            font-weight: bold;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .order-details {
            margin-bottom: 20px;
        }
        
        .detail-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .detail-label {
            color: var(--secondary);
        }
        
        .detail-value {
            font-weight: 500;
        }
        
        .total-amount {
            background: var(--light);
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            margin: 20px 0;
        }
        
        .amount {
            font-size: 2rem;
            font-weight: bold;
            color: var(--primary);
        }
        
        .message {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .processing {
            background: #cce7ff;
            color: #004085;
            border: 1px solid #b3d7ff;
        }
        
        .warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .payment-instructions {
            background: #e7f3ff;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 0.9rem;
        }
        
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3498db;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        
        .support-info {
            background: #e7f3ff;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            font-size: 0.9rem;
        }
        
        .manual-refresh-btn {
            margin-top: 10px;
            padding: 8px 15px;
            background: var(--info);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.9rem;
        }
        
        .manual-refresh-btn:hover {
            background: #138496;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @media (max-width: 768px) {
            .container {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Complete Your Payment</h1>
            <p>Pay for your <?php echo htmlspecialchars($package['name']); ?> VPS</p>
        </div>
        
        <div class="payment-form">
            <?php if ($message && (strpos($message, '⌛ Payment initiated!') !== false || strpos($message, '⌛ Payment processing') !== false)): ?>
                <div class="message processing">
                    <?php if (strpos($message, '⌛ Payment processing') !== false): ?>
                        <div class="loading-spinner"></div>
                    <?php endif; ?>
                    <?php echo $message; ?>
                    
                    <?php if (strpos($message, 'longer than usual') !== false): ?>
                        <div style="margin-top: 15px;">
                            <button onclick="manualRefresh()" class="manual-refresh-btn">
                                🔄 Refresh Now
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php if (strpos($message, 'longer than usual') !== false): ?>
                    <div class="support-info">
                        <strong>📞 Need Help?</strong><br>
                        If your payment is stuck, please contact support with your Transaction ID.<br>
                        <strong>Transaction ID:</strong> <?php echo $_SESSION['payment_check']['transaction_id'] ?? 'N/A'; ?>
                    </div>
                <?php endif; ?>
                
            <?php elseif ($message): ?>
                <div class="message success"><?php echo $message; ?></div>
                <div style="text-align: center; margin-top: 20px;">
                    <a href="dashboard.php" class="btn btn-primary">Go to Dashboard</a>
                    <a href="vps_packages.php" class="btn btn-primary">Order Another VPS</a>
                    <?php if (isAdmin()): ?>
                        <a href="payment_logs.php" class="btn btn-info">View Payment Logs</a>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <?php if ($error): ?>
                    <div class="message error"><?php echo htmlspecialchars($error); ?></div>
                    <div style="text-align: center; margin-top: 15px;">
                        <a href="vps_payment.php" class="btn btn-primary">Try Again</a>
                        <a href="vps_configure.php?package=<?php echo $order['package_id']; ?>" class="btn btn-warning">Back to Configuration</a>
                    </div>
                <?php else: ?>
                    <div class="payment-instructions">
                        <strong>Payment Instructions:</strong><br>
                        1. Enter your Airtel Money phone number<br>
                        2. Click "Process Payment"<br>
                        3. You will receive a prompt on your phone<br>
                        4. Enter your PIN to complete payment<br>
                        5. Wait 15-30 seconds for payment confirmation<br>
                        6. Your VPS will be created automatically after successful payment
                    </div>
                    
                    <form method="POST" id="paymentForm">
                        <div class="form-group">
                            <label for="phone_number">Airtel Money Phone Number *</label>
                            <input type="tel" id="phone_number" name="phone_number" 
                                   placeholder="0991513401" required 
                                   pattern="[0-9]{9,13}" title="Enter your Airtel Money number">
                            <div style="font-size: 0.85rem; color: var(--secondary); margin-top: 5px;">
                                Format: 0991513401 (without +265)
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-success btn-block btn-lg" id="submitBtn">
                            Process Payment
                        </button>
                    </form>
                    
                    <div style="text-align: center; margin-top: 15px;">
                        <a href="vps_configure.php?package=<?php echo $order['package_id']; ?>" class="btn btn-primary">
                            Back to Configuration
                        </a>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <div class="order-summary">
            <div class="package-name"><?php echo htmlspecialchars($package['name']); ?></div>
            
            <div class="order-details">
                <div class="detail-row">
                    <span class="detail-label">Instance Name:</span>
                    <span class="detail-value"><?php echo htmlspecialchars($configuration['display_name']); ?></span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Region:</span>
                    <span class="detail-value"><?php echo htmlspecialchars($configuration['region']); ?></span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Contract Period:</span>
                    <span class="detail-value"><?php echo $configuration['period']; ?> Month(s)</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Operating System:</span>
                    <span class="detail-value">
                        <?php 
                        $os_map = [
                            'afecbb85-e2fc-46f0-9684-b46b1faf00bb' => 'Ubuntu 22.04',
                            '23b6c5c5-39c3-4f80-8fae-2b4fc65b5c5c' => 'Ubuntu 20.04',
                            'd65ecf3b-30db-4dc2-9e88-dfc21a14a6bc' => 'Debian 11',
                            'a1b2c3d4-e5f6-7890-abcd-ef1234567890' => 'CentOS 9',
                            'b2c3d4e5-f6g7-8901-bcde-f23456789012' => 'Windows Server 2022'
                        ];
                        echo $os_map[$configuration['image_id']] ?? 'Custom OS';
                        ?>
                    </span>
                </div>
                <?php if (!empty($configuration['ssh_keys'])): ?>
                <div class="detail-row">
                    <span class="detail-label">SSH Keys:</span>
                    <span class="detail-value"><?php echo count(explode(',', $configuration['ssh_keys'])); ?> key(s)</span>
                </div>
                <?php endif; ?>
            </div>
            
            <div class="total-amount">
                <div>Total Amount</div>
                <div class="amount">
                    <?php 
                    $total_amount = $package['price_monthly'] * $configuration['period'];
                    echo $package['currency'] . number_format($total_amount, 2);
                    ?>
                </div>
                <div style="font-size: 0.9rem; color: var(--secondary);">
                    <?php echo $package['currency'] . $package['price_monthly']; ?> × <?php echo $configuration['period']; ?> month(s)
                </div>
            </div>
            
            <ul style="list-style: none; font-size: 0.9rem; color: var(--secondary); padding-left: 0;">
                <li>✓ <?php echo $package['cpu']; ?> vCPU Cores</li>
                <li>✓ <?php echo $package['ram']; ?> GB RAM</li>
                <li>✓ <?php echo $package['disk']; ?> GB <?php echo $package['disk_type']; ?></li>
                <li>✓ <?php echo $package['traffic']; ?> Traffic</li>
                <?php if ($package['backup']): ?>
                    <li>✓ Free Auto Backup</li>
                <?php endif; ?>
                <?php if ($package['snapshot']): ?>
                    <li>✓ Snapshot Support</li>
                <?php endif; ?>
            </ul>
            
            <?php if (isset($_SESSION['payment_check'])): ?>
            <div class="support-info">
                <strong>📞 Support Information</strong>
                <p style="margin: 10px 0 0 0; font-size: 0.85rem;">
                    If you experience issues, contact support with this Transaction ID:<br>
                    <strong><?php echo $_SESSION['payment_check']['transaction_id']; ?></strong>
                </p>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        document.getElementById('paymentForm')?.addEventListener('submit', function(e) {
            const submitBtn = document.getElementById('submitBtn');
            submitBtn.innerHTML = '<div class="loading-spinner"></div> Processing Payment...';
            submitBtn.disabled = true;
        });
        
        // Format phone number input
        document.getElementById('phone_number')?.addEventListener('input', function(e) {
            this.value = this.value.replace(/[^0-9]/g, '');
        });
        
        // Add manual refresh button functionality
        function manualRefresh() {
            window.location.reload();
        }
        
        // Prevent form resubmission on page refresh
        if (window.history.replaceState) {
            window.history.replaceState(null, null, window.location.href);
        }
    </script>
</body>
</html>